﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_CachedNintendoAccountInfoTypes.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_Types.h>
#include <nn/account/nas/account_Interface.sfdl.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/nas/account_NasTypes.h>

#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_Out.h>

namespace nn { namespace account { namespace detail {
class IAsyncContext;
class IAsyncNetworkServiceLicenseKindContext;
}}} // ~namespace nn::account::detail

namespace nn { namespace account { namespace nas {

template <class Allocator, typename UserRefGetter>
class NintendoAccountExtention
{
private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    NasSessionPool<Allocator>& m_NasSessionPool;
    NasOperator& m_NasOp;
    Executor& m_Executor;

public:
    NintendoAccountExtention(
        Allocator& allocator,
        NasSessionPool<Allocator>& nasSessionPool,
        NasOperator& nasOp,
        Executor& executor) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_NasSessionPool(nasSessionPool)
        , m_NasOp(nasOp)
        , m_Executor(executor)
    {
    }

    Result GetNintendoAccountId(sf::Out<NintendoAccountId> pOutId) NN_NOEXCEPT;

    Result TryRecoverNintendoAccountUserStateAsync(
        sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT;

    Result CreateAuthorizationRequest(
        sf::Out<sf::SharedPointer<nas::IAuthorizationRequest>> pOut,
        const NintendoAccountAuthorizationRequestParameters& param, sf::NativeHandle&& transferMemoryHandle, uint32_t size) NN_NOEXCEPT;

    Result CreateAuthorizationRequest(
        sf::Out<sf::SharedPointer<nas::IAuthorizationRequest>> pOut,
        const nas::NasClientInfo& nasClientInfo, const NintendoAccountAuthorizationRequestParameters& param, sf::NativeHandle&& transferMemoryHandle, uint32_t size) NN_NOEXCEPT;

    Result ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(
        sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT;

    Result RefreshNintendoAccountUserResourceCacheAsync(
        sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT;

    Result RefreshNintendoAccountUserResourceCacheAsyncIfSecondsElapsed(
        sf::Out<bool> pOutMatched, sf::Out<nn::sf::SharedPointer<detail::IAsyncContext>> pOutContext, uint32_t seconds) NN_NOEXCEPT;

    Result GetNintendoAccountUserResourceCache(
        sf::Out<NintendoAccountId> pOutId, sf::Out<NasUserBase> pOutBase,
        const sf::OutBuffer& workBuffer) NN_NOEXCEPT;

    Result GetNintendoAccountUserResourceCacheForApplication(
        sf::Out<NintendoAccountId> pOutId, sf::Out<NasUserBaseForApplication> pOutBase,
        const sf::OutBuffer& workBuffer) NN_NOEXCEPT;

    Result RefreshNetworkServiceLicenseCacheAsync(
        sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT;

    Result RefreshNetworkServiceLicenseCacheAsyncIfSecondsElapsed(
        sf::Out<bool> pOutMatched, sf::Out<nn::sf::SharedPointer<detail::IAsyncContext>> pOutContext, uint32_t seconds) NN_NOEXCEPT;

    Result GetNetworkServiceLicenseCache(
        sf::Out<int32_t> pOutLicense, sf::Out<time::PosixTime> expiration) NN_NOEXCEPT;

    Result LoadNetworkServiceLicenseKindAsync(
        sf::Out<sf::SharedPointer<detail::IAsyncNetworkServiceLicenseKindContext>> pOut) NN_NOEXCEPT;
};

}}} // ~namespace nn::account

// 実装 ---------------------------------------------------------------------------------------

#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/detail/account_AsyncContextImpl.h>
#include <nn/account/nas/account_NintendoAccountAuthorizationRequestImpl.h>

namespace nn { namespace account { namespace nas {

template <typename Task>
class AsyncContext
    : public detail::AsyncTask<Task, ExecutionResource>
{
private:
    user::UserRef m_User;
public:
    AsyncContext(const user::UserRef&& user, NasOperator& nasOp) NN_NOEXCEPT
        : detail::AsyncTask<Task, ExecutionResource>(user, nasOp)
        , m_User(std::move(user))
    {
    }
};

class AsyncNetworkServiceLicenseKindContext
    : public detail::AsyncTask<NetworkServiceLicenseUpdateTask, ExecutionResource>
{
private:
    user::UserRef m_User;
    NasOperator& m_NasOp;
public:
    AsyncNetworkServiceLicenseKindContext(const user::UserRef&& user, NasOperator& nasOp) NN_NOEXCEPT
        : detail::AsyncTask<NetworkServiceLicenseUpdateTask, ExecutionResource>(user, nasOp)
        , m_User(std::move(user))
        , m_NasOp(nasOp)
    {
    }

    Result GetNetworkServiceLicenseKind(sf::Out<int32_t> pOutLicense) NN_NOEXCEPT
    {
        bool hasDone;
        NN_RESULT_DO(HasDone(&hasDone));
        NN_RESULT_THROW_UNLESS(hasDone, ResultInvalidProtocolAccess());
        NN_RESULT_THROW_UNLESS(GetResult().IsSuccess(), ResultInvalidProtocolAccess());

        NetworkServiceLicense license;
        time::PosixTime expiration;
        NN_RESULT_DO(m_NasOp.GetNetworkServiceLicenseCache(&license, &expiration, static_cast<Uid>(m_User)));
        *pOutLicense = static_cast<int32_t>(license);
        NN_RESULT_SUCCESS;
    }
};

#define NN_ACCOUNT_DEFINE_SERVICE_METHOD(rtype, methodInfo) \
template <class Allocator, typename UserRefGetter> \
inline rtype NintendoAccountExtention<Allocator, UserRefGetter>::methodInfo NN_NOEXCEPT

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetNintendoAccountId(sf::Out<NintendoAccountId> pOutId))
{
    NN_RESULT_THROW_UNLESS(pOutId.GetPointer(), ResultNullptr());

    auto user = static_cast<Uid>(static_cast<UserRefGetter*>(this)->GetUserReference());
    // 連携していれば取得できる。
    auto r = this->m_NasOp.CheckAvailability(user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountInvalidState::Includes(r), r);

    return m_NasOp.GetLinkedNintendoAccountId(pOutId.GetPointer(), user);
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    TryRecoverNintendoAccountUserStateAsync(
        sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    auto r = m_NasOp.CheckAvailability(ref);
    // NOTE: NA を利用可能 OR 対話で何とかなる場合が対象
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(r), r);

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, AsyncContext<NasUserStateRecoveryTryTask>>(
        &m_Allocator, static_cast<UserRefGetter*>(this)->GetUserReference(), m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CreateAuthorizationRequest(
        sf::Out<sf::SharedPointer<nas::IAuthorizationRequest>> pOut,
        const NintendoAccountAuthorizationRequestParameters& params, sf::NativeHandle&& transferMemoryHandle, uint32_t size))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    NN_RESULT_DO(m_NasOp.CheckAvailability(ref));

    auto p = Factory::template CreateSharedEmplaced<nas::IAuthorizationRequest, nas::NintendoAccountAuthorizationRequestImpl<Allocator>>(
        &m_Allocator, m_Allocator, ref, m_NasOp, m_NasSessionPool, m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(
        static_cast<UserRefGetter*>(this)->GetApplicationInfo(), params, m_NasOp,
        std::move(transferMemoryHandle), static_cast<size_t>(size)));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CreateAuthorizationRequest(
        sf::Out<sf::SharedPointer<nas::IAuthorizationRequest>> pOut,
        const nas::NasClientInfo& nasClientInfo, const NintendoAccountAuthorizationRequestParameters& params, sf::NativeHandle&& transferMemoryHandle, uint32_t size))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    NN_RESULT_DO(m_NasOp.CheckAvailability(ref));

    auto p = Factory::template CreateSharedEmplaced<nas::IAuthorizationRequest, nas::NintendoAccountAuthorizationRequestImpl<Allocator>>(
        &m_Allocator, m_Allocator, ref, m_NasOp, m_NasSessionPool, m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(
        nasClientInfo, params, m_NasOp,
        std::move(transferMemoryHandle), static_cast<size_t>(size)));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut, const detail::Uuid& sessionId))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    NN_RESULT_DO(m_NasOp.CheckAvailability(ref));

    auto p = Factory::template CreateSharedEmplaced<http::IOAuthProcedure, nas::NintendoAccountApplicationAuthorizationProcedureImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_Executor, ref, m_NasSessionPool);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(sessionId));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    RefreshNintendoAccountUserResourceCacheAsync(sf::Out<nn::sf::SharedPointer<detail::IAsyncContext>> pOut))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    NN_RESULT_DO(m_NasOp.CheckAvailability(ref));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, AsyncContext<NasUserResourceUpdateTask>>(
        &m_Allocator, static_cast<UserRefGetter*>(this)->GetUserReference(), m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    RefreshNintendoAccountUserResourceCacheAsyncIfSecondsElapsed(
        sf::Out<bool> pOutMatched, sf::Out<nn::sf::SharedPointer<detail::IAsyncContext>> pOutContext, uint32_t seconds))
{
    auto user = static_cast<Uid>(static_cast<UserRefGetter*>(this)->GetUserReference());

    TimeSpan sinceLastUpdate;
    auto r = m_NasOp.GetTimeSpanSinceLastUpdateOfUserResourceCache(&sinceLastUpdate, user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultResourceCacheUnavailable::Includes(r), r);
    if (r.IsSuccess() && sinceLastUpdate < TimeSpan::FromSeconds(seconds))
    {
        *pOutMatched = false;
        NN_RESULT_SUCCESS;
    }
    *pOutMatched = true;
    return RefreshNintendoAccountUserResourceCacheAsync(pOutContext);
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetNintendoAccountUserResourceCache(sf::Out<NintendoAccountId> pOutId, sf::Out<NasUserBase> pOutBase, const sf::OutBuffer& workBuffer))
{
    NN_RESULT_THROW_UNLESS(pOutId.GetPointer(), ResultNullptr());
    NN_RESULT_THROW_UNLESS(pOutBase.GetPointer(), ResultNullptr());
    NN_RESULT_THROW_UNLESS(workBuffer.GetPointerUnsafe(), ResultNullptr());
    NN_RESULT_THROW_UNLESS(workBuffer.GetSize() >= RequiredBufferSizeForCachedNintendoAccountInfo, ResultInsufficientBuffer());

    auto user = static_cast<Uid>(static_cast<UserRefGetter*>(this)->GetUserReference());
    // 連携状態が正常か再認証が必要な場合に取得できる。
    auto r = this->m_NasOp.CheckAvailability(user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(r), r);

    NN_RESULT_DO(m_NasOp.LoadUserResourceCache(
        pOutId.GetPointer(), pOutBase.GetPointer(), user,
        workBuffer.GetPointerUnsafe(), workBuffer.GetSize()));
    std::memset(workBuffer.GetPointerUnsafe(), 0xEF, workBuffer.GetSize());
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetNintendoAccountUserResourceCacheForApplication(
        sf::Out<NintendoAccountId> pOutId, sf::Out<NasUserBaseForApplication> pOutBase, const sf::OutBuffer& workBuffer))
{
    NN_RESULT_THROW_UNLESS(pOutId.GetPointer(), ResultNullptr());
    NN_RESULT_THROW_UNLESS(pOutBase.GetPointer(), ResultNullptr());
    NN_RESULT_THROW_UNLESS(workBuffer.GetPointerUnsafe(), ResultNullptr());
    NN_RESULT_THROW_UNLESS(workBuffer.GetSize() >= RequiredBufferSizeForCachedNintendoAccountInfo, ResultInsufficientBuffer());

    auto user = static_cast<Uid>(static_cast<UserRefGetter*>(this)->GetUserReference());
    NN_RESULT_DO(this->m_NasOp.CheckAvailability(user));

    NN_RESULT_DO(m_NasOp.LoadUserResourceCacheForApplication(
        pOutId.GetPointer(), pOutBase.GetPointer(), user,
        workBuffer.GetPointerUnsafe(), workBuffer.GetSize()));
    std::memset(workBuffer.GetPointerUnsafe(), 0xA4, workBuffer.GetSize());
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    RefreshNetworkServiceLicenseCacheAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    NN_RESULT_DO(m_NasOp.CheckAvailability(ref));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, AsyncContext<NetworkServiceLicenseUpdateTask>>(
        &m_Allocator, static_cast<UserRefGetter*>(this)->GetUserReference(), m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    RefreshNetworkServiceLicenseCacheAsyncIfSecondsElapsed(
        sf::Out<bool> pOutMatched, sf::Out<nn::sf::SharedPointer<detail::IAsyncContext>> pOutContext, uint32_t seconds))
{
    auto user = static_cast<Uid>(static_cast<UserRefGetter*>(this)->GetUserReference());

    TimeSpan sinceLastUpdate;
    auto r = m_NasOp.GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(&sinceLastUpdate, user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultResourceCacheUnavailable::Includes(r), r);
    if (r.IsSuccess() && sinceLastUpdate < TimeSpan::FromSeconds(seconds))
    {
        *pOutMatched = false;
        NN_RESULT_SUCCESS;
    }
    *pOutMatched = true;
    return RefreshNetworkServiceLicenseCacheAsync(pOutContext);
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetNetworkServiceLicenseCache(sf::Out<int32_t> pOutLicense, sf::Out<time::PosixTime> pOutExpiration))
{
    auto user = static_cast<Uid>(static_cast<UserRefGetter*>(this)->GetUserReference());
    auto r = this->m_NasOp.CheckAvailability(user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(r), r);

    NetworkServiceLicense license;
    time::PosixTime expiration;
    NN_RESULT_DO(m_NasOp.GetNetworkServiceLicenseCache(&license, &expiration, user));
    *pOutLicense = static_cast<int32_t>(license);
    *pOutExpiration = expiration;
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    LoadNetworkServiceLicenseKindAsync(sf::Out<sf::SharedPointer<detail::IAsyncNetworkServiceLicenseKindContext>> pOut))
{
    auto ref = static_cast<UserRefGetter*>(this)->GetUserReference();
    NN_RESULT_DO(m_NasOp.CheckAvailability(ref));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncNetworkServiceLicenseKindContext, AsyncNetworkServiceLicenseKindContext>(
        &m_Allocator, std::move(ref), m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

#undef NN_ACCOUNT_DEFINE_SERVICE_METHOD

}}} // ~namespace nn::account::nas
