﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_CacheBase.h>
#include <nn/account/http/account_OAuthTypes.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/nas/account_ResultForNas.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/account/account_ResultForAdministrators.h>

#include <utility>

namespace nn {
namespace account {
namespace nas {

typedef detail::CacheSearchResultConfig<ResultNintendoAccountAuthorizationSessionNotExist, ResultNotSupported> NasSessionResultConfig;

struct NasPkceProperty
{
    CallbackUri callback;
    http::CodeVerifier codeVerifier;
    State state;
};

enum PkceSessionTag
{
    PkceSessionTag_Linkage,
    PkceSessionTag_CredentialUpdate,
    PkceSessionTag_NnidLinkage,
};

struct PkceSessionOperator
{
    static bool IsExpired(int64_t expiration) NN_NOEXCEPT;
    static void DeleteCacheData(const std::pair<std::pair<PkceSessionTag, Uid>, detail::Uuid>& data, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
};

class PkceSessionCache
    : detail::CacheBase1<std::pair<std::pair<PkceSessionTag, Uid>, detail::Uuid>, PkceSessionOperator, NasSessionResultConfig>
{
private:
    typedef detail::CacheBase1<std::pair<std::pair<PkceSessionTag, Uid>, detail::Uuid>, PkceSessionOperator, NasSessionResultConfig> Base;

    Result StoreImpl(detail::Uuid* pOut, PkceSessionTag tag, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT;
    Result ExtractImpl(NasPkceProperty* pOutProp, PkceSessionTag tag, const Uid& user, const detail::Uuid& cacheId) NN_NOEXCEPT;

public:
    explicit PkceSessionCache(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(Base::Initialize(storage));
    }

    Result StoreForLinkage(detail::Uuid* pOut, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
    {
        return StoreImpl(pOut, PkceSessionTag_Linkage, user, prop);
    }
    Result ExtractForLinkage(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& cacheId) NN_NOEXCEPT
    {
        return ExtractImpl(pOutProp, PkceSessionTag_Linkage, user, cacheId);
    }

    Result StoreForCredentialUpdate(detail::Uuid* pOut, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
    {
        return StoreImpl(pOut, PkceSessionTag_CredentialUpdate, user, prop);
    }
    Result ExtractForCredentialUpdate(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& cacheId) NN_NOEXCEPT
    {
        return ExtractImpl(pOutProp, PkceSessionTag_CredentialUpdate, user, cacheId);
    }

    Result StoreForNnidLinkage(detail::Uuid* pOut, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
    {
        return StoreImpl(pOut, PkceSessionTag_NnidLinkage, user, prop);
    }
    Result ExtractForNnidLinkage(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& cacheId) NN_NOEXCEPT
    {
        return ExtractImpl(pOutProp, PkceSessionTag_NnidLinkage, user, cacheId);
    }
};

} // ~nn::account::nas
}
}
