﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_Execution.h>
#include <nn/account/http/account_OAuthTypes.h>
#include <nn/account/nas/account_NasLoginDriver.h>
#include <nn/account/nas/account_NasOp2Driver.h>
#include <nn/account/nas/account_NasOp2MembershipCache.h>
#include <nn/account/nas/account_NasSessionCache.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/nas/account_NasUserDriver.h>
#include <nn/account/nas/account_NasUserResourceCache.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_RuntimeResource.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_Result.h>
#include <nn/os/os_TransferMemory.h>

namespace nn { namespace account {

namespace ndas {
class NdasOperator;
} // ~namespace nn::account::ndas

}} // ~namespace nn::account


namespace nn {
namespace account {
namespace nas {

class NasOperator
{
private:
    const NasLoginDriver m_NasLoginDriver;
    const NasUserDriver m_NasUserDriver;
    const NasOp2Driver m_NasOp2Driver;

    ndas::NdasOperator& m_NdasOperator;
    baas::BaasOperator& m_BaasOp;
    NasUserResourceCache& m_NasUserResourceCache;
    NasOp2MembershipCache m_NasOp2MembershipCache;
    const detail::AbstractLocalStorage& m_Storage;

    PkceSessionCache m_PkceSessionCache;

    Result AcrequireAccessToken(
        NasAccessTokenCache *pOut, const Uid& user, const Scope& scope,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

public:
    NasOperator(
        nas::NasUserResourceCache& nasUserResourceCache,
        baas::BaasOperator& baasOperator,
        ndas::NdasOperator& ndasOperator,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;

    const detail::AbstractLocalStorage& GetStorageRef() const NN_NOEXCEPT
    {
        return m_Storage;
    }

    Result CheckAvailability(const Uid& user) const NN_NOEXCEPT;
    Result GetLinkedNintendoAccountId(NintendoAccountId* pOutId, const Uid& user) const NN_NOEXCEPT;
    Result DegradeLinkedNintendoAccountUserState(const Uid& user, baas::NintendoAccountUserState state) NN_NOEXCEPT;

    Result GetRequestForLinkage(
        RequestUrl* pOutRequest,
        const http::CodeVerifier& codeVerifier, const State& state,
        NintendoAccountAuthorizationPageTheme theme) const NN_NOEXCEPT;
    Result Link(
        bool* pOutIsNetworkServiceAccountReplaced,
        const Uid& user, const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result SuspendPkceSessionForLinkage(detail::Uuid* pOutSessionId, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT;
    Result ResumePkceSessionForLinkage(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& sessionId) NN_NOEXCEPT;

    Result GetRequestForCredentialUpdate(
        RequestUrl* pOutRequest,
        const Uid& user, const http::CodeVerifier& codeVerifier, const State& state) const NN_NOEXCEPT;
    Result UpdateCredential(
        const Uid& user, const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result SuspendPkceSessionForCredentialUpdate(detail::Uuid* pOutSessionId, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT;
    Result ResumePkceSessionForCredentialUpdate(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& sessionId) NN_NOEXCEPT;

    Result GetRequestForNnidLinkage(RequestUrl* pOutRequest, const Uid& user, const State& state) NN_NOEXCEPT;
    Result SuspendPkceSessionForNnidLinkage(detail::Uuid* pOutSessionId, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT;
    Result ResumePkceSessionForNnidLinkage(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& sessionId) NN_NOEXCEPT;

    Result GetClientInfo(
        NasClientInfo* pOut, const detail::ApplicationInfo& appInfo,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetRequestForApplicationAuthorizationRequest(
        RequestUrl* pOutRequest,
        const Uid& user,
        const NasClientInfo& clientInfo, const Scope& scope, const State& state, const Nonce& nonce) const NN_NOEXCEPT;
    Result AcquireApplicationAuthorizationWithoutPrompt(
        NasApplicationAuthorization* pOut,
        const Uid& user, const NasClientInfo& clientInfo,
        const Scope& scope, const State& state, const Nonce& nonce,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result GetRequestForGuestLogin(RequestUrl* pOutRequest, const State& state) const NN_NOEXCEPT;
    Result GetCredentialFromAuthorizationCode(
        NintendoAccountId* pOutNaId, NasCredentialCache *pOutCredentialCache,
        const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result UpdateUserResourceCache(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetTimeSpanSinceLastUpdateOfUserResourceCache(TimeSpan* pOut, const Uid& user) NN_NOEXCEPT;
    Result LoadUserResourceCache(NintendoAccountId* pOutId, NasUserBase* pOutBase, const Uid& user, void* rawBuffer, size_t bufferSize) const NN_NOEXCEPT;
    Result LoadUserResourceCacheForApplication(NintendoAccountId* pOutId, NasUserBaseForApplication* pOutBase, const Uid& user, void* rawBuffer, size_t bufferSize) const NN_NOEXCEPT;
    void InvalidateUserResourceCacheTimestamp(const Uid& user) NN_NOEXCEPT;

    Result UpdateNetworkServiceLicenseCache(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(TimeSpan* pOut, const Uid& user) NN_NOEXCEPT;
    Result GetNetworkServiceLicenseCache(NetworkServiceLicense* pOutLicense, time::PosixTime* pOutExpiration, const Uid& user) const NN_NOEXCEPT;
    void InvalidateNetworkServiceLicenseCache(const Uid& user) NN_NOEXCEPT;

    Result TryRecoverNintendoAccountUserStateImplicitly(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

class NasUserStateRecoveryTryTask
    : public detail::Executable<ExecutionResource>
{
    NN_DISALLOW_COPY(NasUserStateRecoveryTryTask);
private:
    NasOperator& m_NasOp;
    const Uid m_User;

protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_NasOp.TryRecoverNintendoAccountUserStateImplicitly(m_User, resource, this);
    }
public:
    NasUserStateRecoveryTryTask(const Uid& user, NasOperator& nasOp)
        : m_NasOp(nasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(user);
    }
};

class NasUserResourceUpdateTask
    : public detail::Executable<ExecutionResource>
{
    NN_DISALLOW_COPY(NasUserResourceUpdateTask);
private:
    NasOperator& m_NasOp;
    const Uid m_User;

protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_NasOp.UpdateUserResourceCache(m_User, resource, this);
    }
public:
    NasUserResourceUpdateTask(const Uid& user, NasOperator& nasOp)
        : m_NasOp(nasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(user);
    }
};

class NetworkServiceLicenseUpdateTask
    : public detail::Executable<ExecutionResource>
{
    NN_DISALLOW_COPY(NetworkServiceLicenseUpdateTask);
private:
    NasOperator& m_NasOp;
    const Uid m_User;

protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_NasOp.UpdateNetworkServiceLicenseCache(m_User, resource, this);
    }
public:
    NetworkServiceLicenseUpdateTask(const Uid& user, NasOperator& nasOp)
        : m_NasOp(nasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(user);
    }
};

class NasLinkageProcedure
{
    NN_DISALLOW_COPY(NasLinkageProcedure);

private:
    NasOperator& m_NasOp;
    const Uid m_User;
    http::ProcedureState m_ProcState;

    // リクエストパラメータ
    NasPkceProperty m_Property;

    // 受信データ
    detail::Uuid m_CodeCacheId;
    bool m_IsNetworkServiceAccountReplaced;

public:
    NasLinkageProcedure(const Uid& user, NasOperator& nasOp) NN_NOEXCEPT;
    ~NasLinkageProcedure() NN_NOEXCEPT;

    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri, NintendoAccountAuthorizationPageTheme theme) NN_NOEXCEPT;
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;
    Result ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result Suspend(detail::Uuid* pOutSessionId) NN_NOEXCEPT;
    Result Resume(const detail::Uuid& sessionId) NN_NOEXCEPT;

    bool IsNetworkServiceAccountReplaced() const NN_NOEXCEPT
    {
        return m_IsNetworkServiceAccountReplaced;
    }
};

class NasCredentialUpdateProcedure
{
    NN_DISALLOW_COPY(NasCredentialUpdateProcedure);

private:
    NasOperator& m_NasOp;
    const Uid m_User;
    http::ProcedureState m_ProcState;

    // リクエストパラメータ
    NasPkceProperty m_Property;

    // 受信データ
    detail::Uuid m_CodeCacheId;

public:
    NasCredentialUpdateProcedure(const Uid& user, NasOperator& nasOp) NN_NOEXCEPT;
    ~NasCredentialUpdateProcedure() NN_NOEXCEPT;

    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT;
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;
    Result ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result Suspend(detail::Uuid* pOutSessionId) NN_NOEXCEPT;
    Result Resume(const detail::Uuid& sessionId) NN_NOEXCEPT;
};

class NasNnidLinkageProcedure
{
    NN_DISALLOW_COPY(NasNnidLinkageProcedure);

private:
    NasOperator& m_NasOp;
    const Uid m_User;
    http::ProcedureState m_ProcState;

    // リクエストパラメータ
    NasPkceProperty m_Property;

public:
    NasNnidLinkageProcedure(const Uid& user, NasOperator& nasOp) NN_NOEXCEPT;

    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT;
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;
    Result ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result Suspend(detail::Uuid* pOutSessionId) NN_NOEXCEPT;
    Result Resume(const detail::Uuid& sessionId) NN_NOEXCEPT;
};

class NasApplicationAuthorizationProcedure
{
    NN_DISALLOW_COPY(NasApplicationAuthorizationProcedure);

private:
    NasOperator& m_NasOp;
    const Uid m_User;
    http::ProcedureState m_ProcState;

    // リクエストパラメータ
    const detail::ApplicationInfo m_AppInfo;
    NasClientInfo m_NasClientInfo;

    // リソース
    bool m_Mapped;
    os::TransferMemory m_Memory;
    struct Resource
    {
        // リクエストパラメータ
        Scope scope;
        State state;
        Nonce nonce;

        // 受信
        NasApplicationAuthorization authorization;
    }* m_pResource;
    NN_STATIC_ASSERT(sizeof(Resource) <= 4096);

public:
    static const size_t RequiredMemorySize = sizeof(Resource);

    NasApplicationAuthorizationProcedure(const Uid& user, const NasClientInfo& clientInfo, NasOperator& nasOp) NN_NOEXCEPT;
    NasApplicationAuthorizationProcedure(const Uid& user, const detail::ApplicationInfo& appInfo, NasOperator& nasOp) NN_NOEXCEPT;
    ~NasApplicationAuthorizationProcedure() NN_NOEXCEPT;

    void AttachTransferMemory(os::NativeHandle transferMemoryHandle, size_t size, bool managed) NN_NOEXCEPT;
    Result Initialize(const NintendoAccountAuthorizationRequestParameters& params) NN_NOEXCEPT;

    Result ExecuteImplicitly(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result ExecutePreprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) const NN_NOEXCEPT;
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;

    bool IsAuthorized() const NN_NOEXCEPT;
    Result GetAuthorizationCode(size_t* pOutActualSize, char* buffer, size_t size) const NN_NOEXCEPT;
    Result GetIdToken(size_t* pOutActualSize, char* buffer, size_t size) const NN_NOEXCEPT;
    Result GetState(State* pOut) const NN_NOEXCEPT;
};

class NasGuestLoginProcedure
{
    NN_DISALLOW_COPY(NasGuestLoginProcedure);

private:
    ndas::NdasOperator& m_NdasOp;
    baas::BaasOperator& m_BaasOp;
    NasOperator& m_NasOp;
    http::ProcedureState m_ProcState;
    const detail::ApplicationInfo m_AppInfo;

    // リソース
    bool m_Mapped;
    os::TransferMemory m_Memory;
    struct Resource
    {
        // リクエストパラメータ
        CallbackUri callbackUri;
        State state;
        char nasIdToken[detail::NasIdTokenSizeMax];

        // 受信
        baas::GuestUserProfile userInfo;
        size_t imageSize;
        char image[ProfileImageBytesMax];
    }* m_pResource;
    NN_STATIC_ASSERT(sizeof(Resource) <= 4096 * (2 + 32)); // 136KiB 要る

public:
    static const size_t RequiredMemorySize = sizeof(Resource);

    NasGuestLoginProcedure(
        const detail::ApplicationInfo& appInfo,
        ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT;
    ~NasGuestLoginProcedure() NN_NOEXCEPT;

    void AttachTransferMemory(os::NativeHandle transferMemoryHandle, size_t size, bool managed) NN_NOEXCEPT;
    Result Initialize() NN_NOEXCEPT;

    Result ExecutePreprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT;
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;
    Result ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOut) const NN_NOEXCEPT;
    Result GetLinkedNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT;
    Result GetNickname(char* buffer, size_t size) const NN_NOEXCEPT;
    Result GetProfileImage(size_t* pOutActualSize, void* buffer, size_t size) const NN_NOEXCEPT;
    Result LoadBaasIdTokenCache(size_t* pOutActualSize, char* buffer, size_t size) const NN_NOEXCEPT;
};

class NasFloatingRegistrationProcedure
{
    NN_DISALLOW_COPY(NasFloatingRegistrationProcedure);

private:
    const ndas::NdasOperator& m_NdasOp;
    baas::BaasOperator& m_BaasOp;
    NasOperator& m_NasOp;
    http::ProcedureState m_ProcState;

    // リクエストパラメータ
    NasPkceProperty m_Property;
    detail::Uuid m_CodeCacheId;

    // NA
    NintendoAccountId m_NaId;
    NasCredentialCache m_CredentialCache;

    // NSA
    NetworkServiceAccountId m_NsaId;
    baas::BaasCredential m_BaasCredential;

    // リソース
    bool m_Mapped;
    os::TransferMemory m_Memory;
    struct Resource
    {
        baas::UserProfile userInfo;
        size_t imageSize;
        char image[ProfileImageBytesMax];
    }*m_pResource;
    NN_STATIC_ASSERT(sizeof(Resource) <= 4096 * (2 + 32)); // 136KiB 要る

public:
    static const size_t RequiredMemorySize = sizeof(Resource);

    NasFloatingRegistrationProcedure(const ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT;
    ~NasFloatingRegistrationProcedure() NN_NOEXCEPT;

    void AttachTransferMemory(os::NativeHandle transferMemoryHandle, size_t size, bool managed) NN_NOEXCEPT;
    Result Initialize() NN_NOEXCEPT;

    Result RegisterNetworkServiceAccount(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    const baas::UserProfile& GetUserProfileRef() const NN_NOEXCEPT;
    const void* GetProfileImagePtr(size_t* pOut) const NN_NOEXCEPT;

    Result ExecutePreprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT;
    Result ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT;
    Result ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOut) const NN_NOEXCEPT;
    Result GetLinkedNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT;
    Result GetNickname(char* buffer, size_t size) const NN_NOEXCEPT;
    Result GetProfileImage(size_t* pOutActualSize, void* buffer, size_t size) const NN_NOEXCEPT;

    Result EnsureNetworkServiceAccountIdToken(const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result LoadBaasIdTokenCache(size_t* pOutActualSize, char* buffer, size_t size, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT;
};

} // ~namespace nn::account::nas
}
}
