﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/http/json/http_JsonPath.h>

namespace nn { namespace account { namespace json {

template <int NodeDepthMax, int PathLengthMax, bool FailOnDepthExcess = true>
using JsonPath = nn::http::json::JsonPath<NodeDepthMax, PathLengthMax, FailOnDepthExcess>;

/** @brief JSONパス文字列の検索用ユーティリティ

    @details
        あるパス文字列で表されるJonPathがこれまで渡されたかどうかを保持する
 */
struct LookupEntry
{
    const char* path;
    bool found;

    explicit LookupEntry(const char* p) NN_NOEXCEPT
        : path(p)
        , found(false)
    {
        NN_SDK_ASSERT(p);
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }

    template <typename JsonPathType>
    bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return !(*this) && jsonPath.Match(path);
    }

    void MarkAccepted() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!found);
        found = true;
    }
};
struct LookupEntry1
{
    const char* path0;
    const char* path1;
    bool found;

    LookupEntry1(const char* p0, const char* p1) NN_NOEXCEPT
        : path0(p0)
        , path1(p1)
        , found(false)
    {
        NN_SDK_ASSERT(p0);
        NN_SDK_ASSERT(p1);
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }

    template <typename JsonPathType>
    bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return !(*this) && jsonPath.Match(path0, path1);
    }

    void MarkAccepted() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!found);
        found = true;
    }
};

/*
    次の条件をすべて満たす場合にのみ AdaptImpl() が呼ばれる。

    - BaseAdaptor に指定したアダプタの Adapt() が ResultSuccess を返す
    - SetIoResult() に ResultSuccess 以外の値が指定されていない
 */
template <typename BaseAdaptor>
class ExtensibleJsonAdaptorBase
    : BaseAdaptor
{
private:
    Result m_IoResult;
    bool CanContinue() const NN_NOEXCEPT;

public:
    typedef typename BaseAdaptor::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT = 0;
    virtual bool UpdateImpl(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
    {
        return false;
    }
    virtual bool UpdateImpl(const JsonPathType&, bool) NN_NOEXCEPT
    {
        return false;
    }
    virtual bool UpdateImpl(const JsonPathType&, int64_t) NN_NOEXCEPT
    {
        return false;
    }
    virtual bool UpdateImpl(const JsonPathType&, const char*, int) NN_NOEXCEPT
    {
        return false;
    }

    Result SetIoResult(Result r) NN_NOEXCEPT;

public:
    ExtensibleJsonAdaptorBase() NN_NOEXCEPT;
    Result Adapt(int32_t httpCode) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT;
    void Update(const JsonPathType&, bool) NN_NOEXCEPT;
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT;
    void Update(const JsonPathType&, double) NN_NOEXCEPT;
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}; // NOP
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}; // NOP
};

}}} // ~namespace nn::account::json

/* --------------------------------------------------------------------------------------------
    実装
 */
#include <cstring>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace json {
template <typename BaseAdaptor>
inline ExtensibleJsonAdaptorBase<BaseAdaptor>::ExtensibleJsonAdaptorBase() NN_NOEXCEPT
    : m_IoResult(ResultSuccess())
{
}
template <typename BaseAdaptor>
inline Result ExtensibleJsonAdaptorBase<BaseAdaptor>::SetIoResult(Result r) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IoResult.IsSuccess());
    m_IoResult = r;
    return m_IoResult;
}
template <typename BaseAdaptor>
inline bool ExtensibleJsonAdaptorBase<BaseAdaptor>::CanContinue() const NN_NOEXCEPT
{
    return m_IoResult.IsSuccess();
}
template <typename BaseAdaptor>
inline Result ExtensibleJsonAdaptorBase<BaseAdaptor>::Adapt(int32_t httpCode) NN_NOEXCEPT
{
    NN_RESULT_DO(m_IoResult);
    NN_RESULT_DO(BaseAdaptor::Adapt(httpCode));
    NN_RESULT_DO(AdaptImpl());
    NN_RESULT_SUCCESS;
}
template <typename BaseAdaptor>
inline void ExtensibleJsonAdaptorBase<BaseAdaptor>::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (!CanContinue()
        || UpdateImpl(jsonPath, value))
    {
        return;
    }
    BaseAdaptor::Update(jsonPath, value);
}
template <typename BaseAdaptor>
inline void ExtensibleJsonAdaptorBase<BaseAdaptor>::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (!CanContinue()
        || UpdateImpl(jsonPath, value, valueLength))
    {
        return;
    }
    BaseAdaptor::Update(jsonPath, value, valueLength);
}
template <typename BaseAdaptor>
inline void ExtensibleJsonAdaptorBase<BaseAdaptor>::Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
{
    if (!CanContinue()
        || UpdateImpl(jsonPath, value))
    {
        return;
    }
    BaseAdaptor::Update(jsonPath, value);
}
template <typename BaseAdaptor>
inline void ExtensibleJsonAdaptorBase<BaseAdaptor>::Update(const JsonPathType& jsonPath, std::nullptr_t value) NN_NOEXCEPT
{
    if (!CanContinue()
        || UpdateImpl(jsonPath, value))
    {
        return;
    }
    BaseAdaptor::Update(jsonPath, value);
}
template <typename BaseAdaptor>
inline void ExtensibleJsonAdaptorBase<BaseAdaptor>::Update(const JsonPathType&p, uint64_t v) NN_NOEXCEPT
{
    if (!CanContinue())
    {
        return;
    }
    BaseAdaptor::Update(p, v);
}
template <typename BaseAdaptor>
inline void ExtensibleJsonAdaptorBase<BaseAdaptor>::Update(const JsonPathType& p, double v) NN_NOEXCEPT
{
    if (!CanContinue())
    {
        return;
    }
    BaseAdaptor::Update(p, v);
}

}}} // ~namespace nn::account::json
