﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/detail/account_LockGuard.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace account { namespace detail {

struct FsLockGuardTag {};
using FsLockGuard = LockGuard<os::SdkRecursiveMutex, FsLockGuardTag>;

class AbstractFileSystem
{
public:
    virtual Result Create(const char* path, size_t sizeToReserve) const NN_NOEXCEPT = 0;
    virtual Result Delete(const char* path) const NN_NOEXCEPT = 0;
    virtual Result Write(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT = 0;
    virtual Result Append(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT = 0;
    virtual Result GetSize(size_t* pOutSize, const char* path) const NN_NOEXCEPT = 0;
    virtual Result SetSize(const char* path, size_t size) const NN_NOEXCEPT = 0;
    virtual Result Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT = 0;
    virtual Result Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path) const NN_NOEXCEPT = 0;
    virtual Result Move(const char* to, const char* from) const NN_NOEXCEPT = 0;
    virtual Result Copy(const char* to, const char* from, void* buffer, size_t bufferSize) const NN_NOEXCEPT = 0;
};

// nn::fs のラッパー
class DefaultFileSystem : public AbstractFileSystem
{
private:
    static os::SdkRecursiveMutex s_WriterLock;

    static Result DeleteDirectoryImpl(const char* path) NN_NOEXCEPT;
    static Result CreateDirectoryImpl(const char* path) NN_NOEXCEPT;
    static Result RenewDirectoryImpl(const char* path) NN_NOEXCEPT;
    static Result DeleteFileImpl(const char* path) NN_NOEXCEPT;
    static Result GetFileSizeImpl(size_t* pOutSize, const char* path) NN_NOEXCEPT;
    static Result SetFileSizeImpl(const char* path, size_t size) NN_NOEXCEPT;
    static Result ReadFileImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) NN_NOEXCEPT;
    static Result WriteFileImpl(const char* path, const void* buffer, size_t bufferSize) NN_NOEXCEPT;
    static Result CreateFileImpl(const char* path, size_t sizeToReserve) NN_NOEXCEPT;
    static Result AppendFileImpl(const char* path, size_t offset, const void* buffer, size_t bufferSize) NN_NOEXCEPT;
    static Result MoveFileImpl(const char* to, const char* from) NN_NOEXCEPT;
    static Result CopyFileImpl(const char* to, const char* from, void* buffer, size_t bufferSize) NN_NOEXCEPT;

public:
    // マウント, アンマウント
    static Result MountHost(const char* volumeName, const char* hostLocation) NN_NOEXCEPT;
    static Result MountSave(const char* volumeName, fs::SystemSaveDataId saveDataId, int64_t dataSize, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;
    static void Unmount(const char* volumeName) NN_NOEXCEPT;
    static Result Commit(const char* volumeName) NN_NOEXCEPT;

    // ディレクトリの操作
    static Result Setup(const char* rootPath) NN_NOEXCEPT;
    static void ClearProfile(const char* rootPath) NN_NOEXCEPT;
    static void ClearCache(const char* rootPath) NN_NOEXCEPT;
    static void Clear(const char* rootPath) NN_NOEXCEPT;

    // 書き込み排他
    static FsLockGuard AcquireWriterLock() NN_NOEXCEPT;

public:
    // ファイルの操作
    virtual Result Create(const char* path, size_t sizeToReserve) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Delete(const char* path) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Append(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetSize(size_t* pOutSize, const char* path) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetSize(const char* path, size_t size) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Move(const char* to, const char* from) const NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Copy(const char* to, const char* from, void* buffer, size_t bufferSize) const NN_NOEXCEPT NN_OVERRIDE;
};

}}} // ~namespace nn::account::detail

#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace detail {

/* --------------------------------------------------------------------------------------------
    FileSystemBase 部分の実装
    */
inline Result DefaultFileSystem::MountHost(const char* volumeName, const char* src) NN_NOEXCEPT
{
    return fs::MountHost(volumeName, src);
}
inline Result DefaultFileSystem::MountSave(const char* volumeName, fs::SystemSaveDataId saveId, int64_t dataSize, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    NN_RESULT_TRY(fs::MountSystemSaveData(volumeName, saveId))
    NN_RESULT_CATCH(fs::ResultTargetNotFound)
    {
        NN_RESULT_DO(fs::CreateSystemSaveData(saveId, dataSize, journalSize, flags));
        NN_RESULT_DO(fs::MountSystemSaveData(volumeName, saveId));
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}
inline void DefaultFileSystem::Unmount(const char* volumeName) NN_NOEXCEPT
{
    fs::Unmount(volumeName);
}
inline Result DefaultFileSystem::Delete(const char* path) const NN_NOEXCEPT
{
    return DeleteFileImpl(path);
}
inline Result DefaultFileSystem::GetSize(size_t* pOutSize, const char* path) const NN_NOEXCEPT
{
    return GetFileSizeImpl(pOutSize, path);
}
inline Result DefaultFileSystem::SetSize(const char* path, size_t size) const NN_NOEXCEPT
{
    return SetFileSizeImpl(path, size);
}
inline Result DefaultFileSystem::Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT
{
    return ReadFileImpl(pOutActualSize, buffer, bufferSize, path, offset);
}
inline Result DefaultFileSystem::Read(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path) const NN_NOEXCEPT
{
    return ReadFileImpl(pOutActualSize, buffer, bufferSize, path, 0u);
}
inline Result DefaultFileSystem::Write(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return WriteFileImpl(path, buffer, bufferSize);
}
inline Result DefaultFileSystem::Create(const char* path, size_t sizeToReserve) const NN_NOEXCEPT
{
    return CreateFileImpl(path, sizeToReserve);
}
inline Result DefaultFileSystem::Append(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return AppendFileImpl(path, offset, buffer, bufferSize);
}
inline Result DefaultFileSystem::Move(const char* to, const char* from) const NN_NOEXCEPT
{
    return DefaultFileSystem::MoveFileImpl(to, from);
}
inline Result DefaultFileSystem::Copy(const char* to, const char* from, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return DefaultFileSystem::CopyFileImpl(to, from, buffer, bufferSize);
}

}}} // ~namespace nn::account::detail
