﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/util/util_Execution.h>

namespace nn { namespace account { namespace detail {

using Cancellable = util::Cancelable;
using CancelInjectable = util::CancelInjectable;

// 定めし手続きで実行 (とキャンセル) 可能な概念
template <typename ExecutionResourceType>
class Executable
    : public util::Executable
{
    NN_DISALLOW_COPY(Executable);
    NN_DISALLOW_MOVE(Executable);

private:
    using Base = util::Executable;

protected:
    Executable() NN_NOEXCEPT = default;

    // 各実装にて内部を記述される
    virtual Result ExecuteImpl(const ExecutionResourceType& resource) NN_NOEXCEPT = 0;

    virtual Result ExecuteImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_ABORT_UNLESS(bufferSize == sizeof(ExecutionResourceType));
        NN_ABORT_UNLESS(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<ExecutionResourceType>::value == 0);
        return ExecuteImpl(*reinterpret_cast<ExecutionResourceType*>(buffer));
    }

public:
    Result GetResult() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(Base::HasDone());
        auto pR = util::Executable::TryGetResult();
        NN_RESULT_THROW_UNLESS(pR, ResultCancelled());
        return *pR;
    }
};

template <int Capacity>
struct ExecutorConfig
{
    static const int QueueCapacity = Capacity;
    using ResultOutOfQueueCapacity = ::nn::account::ResultOutOfQueueCapacity;
};

template <int Capacity, typename ExecutionResourceType>
class Executor
    : public util::Executor<ExecutorConfig<Capacity>>
{
private:
    using Base = util::Executor<ExecutorConfig<Capacity>>;

public:
    void Execute(const ExecutionResourceType& resource) NN_NOEXCEPT
    {
        ExecutionResourceType tmp = resource;
        Base::Execute(&tmp, sizeof(tmp));
    }
};

}}} // ~namespace nn::account::detail
