﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_AsyncContextImpl.h>
#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace account { namespace baas {

template <typename Allocator>
class GuestLoginRequestWithNintendoAccountImpl
{
    NN_DISALLOW_COPY(GuestLoginRequestWithNintendoAccountImpl);

private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    nas::NasSessionPool<Allocator>& m_NasSessionPool;
    Executor& m_Executor;

    detail::Uuid m_SessionId;
    nas::NasGuestLoginProcedure* m_pProcedure;

public:
    GuestLoginRequestWithNintendoAccountImpl(
        Allocator& allocator,
        nas::NasSessionPool<Allocator>& nasSessionPool,  Executor& executor) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_NasSessionPool(nasSessionPool)
        , m_Executor(executor)
        , m_SessionId(detail::InvalidUuid)
        , m_pProcedure(nullptr)
    {
    }
    ~GuestLoginRequestWithNintendoAccountImpl() NN_NOEXCEPT
    {
        if (m_SessionId)
        {
            m_NasSessionPool.ReleaseNasGuestLoginSession(m_SessionId);
        }
    }

    Result Initialize(
        const detail::ApplicationInfo& appInfo, ndas::NdasOperator& ndasOp, BaasOperator& baasOp, nas::NasOperator& nasOp,
        sf::NativeHandle&& transferMemoryHandle, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
        NN_RESULT_THROW_UNLESS(size >= nas::NasGuestLoginProcedure::RequiredMemorySize, ResultInsufficientBuffer());
        NN_RESULT_THROW_UNLESS(!m_SessionId, ResultInvalidProtocolAccess());

        NN_RESULT_DO(m_NasSessionPool.CreateNasGuestLoginSession(&m_SessionId, &m_pProcedure, appInfo, ndasOp, baasOp, nasOp));
        m_pProcedure->AttachTransferMemory(transferMemoryHandle.GetOsHandle(), size, transferMemoryHandle.IsManaged());
        transferMemoryHandle.Detach();

        NN_RESULT_DO(m_pProcedure->Initialize());
        NN_RESULT_SUCCESS;
    }
    Result InvokeWithoutInteractionAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>>) NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    Result GetSessionId(sf::Out<detail::Uuid> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        *pOut = m_SessionId;
        NN_RESULT_SUCCESS;
    }

    Result LoadIdTokenCache(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOut) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->LoadBaasIdTokenCache(&sizeActual, pOut.GetPointerUnsafe(), pOut.GetSize()));
        NN_SDK_ASSERT(sizeActual <= detail::BaasIdTokenSizeMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
    Result GetAccountId(sf::Out<NetworkServiceAccountId> pOutId) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetNetworkServiceAccountId(pOutId.GetPointer());
    }
    Result GetLinkedNintendoAccountId(sf::Out<NintendoAccountId> pOut) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetLinkedNintendoAccountId(pOut.GetPointer());
    }
    Result GetNickname(const sf::OutArray<char>& nickname) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        return m_pProcedure->GetNickname(nickname.GetData(), nickname.GetLength());
    }
    Result GetProfileImage(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOutImage) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_SessionId, ResultInvalidProtocolAccess());

        size_t sizeActual;
        NN_RESULT_DO(m_pProcedure->GetProfileImage(&sizeActual, pOutImage.GetPointerUnsafe(), pOutImage.GetSize()));
        NN_SDK_ASSERT(sizeActual <= ProfileImageBytesMax);
        *pOutActualSize = static_cast<uint32_t>(sizeActual);
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namespace nn::account::baas
