﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasLoginCache.h>
#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/detail/account_Execution.h>
#include <nn/account/detail/account_IAsyncContext.sfdl.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/http/account_Interface.sfdl.h>
#include <nn/account/nas/account_Interface.sfdl.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NintendoAccountProcedureImpl.h>
#include <nn/account/user/account_UserReference.h>
#include <nn/account/user/account_UserRegistry.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_Result.h>
#include <nn/npns/npns_Types.h>
#include <nn/os/os_Types.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account {
class Executor;
}} // ~namespace nn::account

namespace nn { namespace account { namespace baas {

template <typename Allocator>
class BaasAccessTokenAccessorImpl
    : public sf::IServiceObject
{
private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    const user::UserRegistry& m_Registry;
    BaasOperator& m_BaasOp;
    Executor& m_Executor;

public:
    BaasAccessTokenAccessorImpl(
        Allocator& allocator,
        const user::UserRegistry& registry,
        BaasOperator& baasOp,
        Executor& executor) NN_NOEXCEPT;
    Result EnsureCacheAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& uid) NN_NOEXCEPT;
    Result LoadCache(sf::Out<std::uint32_t> pOutActualSize, const sf::OutBuffer& pOut, const Uid& uid) NN_NOEXCEPT;
    Result GetDeviceAccountId(sf::Out<std::uint64_t> pOut, const Uid& uid) NN_NOEXCEPT;

    Result RegisterNotificationTokenAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& uid, const npns::NotificationToken& nt) NN_NOEXCEPT;
    Result UnregisterNotificationTokenAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& uid) NN_NOEXCEPT;
};

template <typename Allocator>
class NetworkServiceAccountAdministratorImpl
{
private:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    BaasOperator& m_BaasOp;
    nas::NasOperator& m_NasOp;
    const user::UserRef m_User;
    Executor& m_Executor;

    detail::ApplicationInfo m_AppInfo;

    Result CreateProcedureToLinkWithNintendoAccountImpl(
        sf::EmplacedRef<nas::IOAuthProcedureForNintendoAccountLinkage, nas::NintendoAccountLinkageProcedureImpl<Allocator>>* pOut) NN_NOEXCEPT;
    Result CreateProcedureToUpdateLinkageStateOfNintendoAccountImpl(
        sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountCredentialUpdateProcedureImpl<Allocator>>* pOut) NN_NOEXCEPT;
    Result CreateProcedureToLinkNnidWithNintendoAccountImpl(
        sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountNnidLinkageProcedureImpl<Allocator>>* pOut) NN_NOEXCEPT;

public:
    NetworkServiceAccountAdministratorImpl(
        Allocator& allocator,
        BaasOperator& baasOp,
        nas::NasOperator& nasOp,
        user::UserRef&& user,
        Executor& executor) NN_NOEXCEPT;
    user::UserRef GetUserReference() const NN_NOEXCEPT
    {
        return m_User;
    }
    detail::ApplicationInfo GetApplicationInfo() const NN_NOEXCEPT
    {
        return m_AppInfo;
    }

    Result SetApplicationInfo(const detail::ApplicationInfo& appInfo) NN_NOEXCEPT;
    Result SetSystemProgramIdentification(const SystemProgramIdentification& identification, const Bit64& pid) NN_NOEXCEPT;
    Result GetServiceEntryRequirementCache(sf::Out<uint32_t> pOut, const ApplicationId& appId) NN_NOEXCEPT;
    Result InvalidateServiceEntryRequirementCache(const ApplicationId& appId) NN_NOEXCEPT;
    Result InvalidateTokenCache(const ApplicationId& appId) NN_NOEXCEPT;

    Result IsRegistered(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result RegisterAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;
    Result UnregisterAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;
    Result DeleteRegistrationInfoLocally() NN_NOEXCEPT;
    Result CheckAvailability() NN_NOEXCEPT;
    Result GetAccountId(sf::Out<account::NetworkServiceAccountId> pOutId) NN_NOEXCEPT;

    Result EnsureIdTokenCacheAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;
    Result LoadIdTokenCache(sf::Out<std::uint32_t> pOutActualSize, const sf::OutBuffer& pOut) NN_NOEXCEPT;

    Result SynchronizeProfileAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;
    Result SynchronizeProfileAsyncIfSecondsElapsed(sf::Out<bool> pOutMatched, sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext, uint32_t seconds) NN_NOEXCEPT;
    Result UploadProfileAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;

    Result IsLinkedWithNintendoAccount(sf::Out<bool> pOut) NN_NOEXCEPT;
    Result CreateProcedureToLinkWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage>> pOut) NN_NOEXCEPT;
    Result ResumeProcedureToLinkWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT;
    Result CreateProcedureToUpdateLinkageStateOfNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut) NN_NOEXCEPT;
    Result ResumeProcedureToUpdateLinkageStateOfNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT;
    Result CreateProcedureToLinkNnidWithNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut) NN_NOEXCEPT;
    Result ResumeProcedureToLinkNnidWithNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT;

    Result DebugUnlinkNintendoAccountAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;
    Result DebugSetAvailabilityErrorDetail(uint32_t error) NN_NOEXCEPT;
};

}}} // ~namespace baas

/* --------------------------------------------------------------------------------------------
    実装
*/
#include <nn/account/account_RuntimeResource.h>
#include <nn/account/detail/account_AsyncContextImpl.h>

namespace nn { namespace account { namespace baas {

/* ------------------------------------------------------------
    非同期タスクの定義
*/
class BaasAccessTokenContext
    : public detail::AsyncTask<BaasLoginTaskForAccessToken, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasAccessTokenContext(user::UserRef&& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasLoginTaskForAccessToken, ExecutionResource>(user, baasOp)
        , m_User(std::move(user))
    {
    }
};
class BaasNotificationRegistrationContext
    : public detail::AsyncTask<BaasNotificationRegistrationTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasNotificationRegistrationContext(user::UserRef&& user, const npns::NotificationToken& nt, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasNotificationRegistrationTask, ExecutionResource>(user, nt, baasOp)
        , m_User(std::move(user))
    {
        NN_SDK_ASSERT(m_User.IsValid());
    }
};
class BaasNotificationErasureContext
    : public detail::AsyncTask<BaasNotificationErasureTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasNotificationErasureContext(user::UserRef&& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasNotificationErasureTask, ExecutionResource>(user, baasOp)
        , m_User(std::move(user))
    {
        NN_SDK_ASSERT(m_User.IsValid());
    }
};

class BaasRegistrationContext
    : public detail::AsyncTask<BaasRegistrationTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasRegistrationContext(const user::UserRef& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasRegistrationTask, ExecutionResource>(user, baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(static_cast<Uid>(user));
    }
};

class BaasErasureContext
    : public detail::AsyncTask<BaasErasureTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasErasureContext(const user::UserRef& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasErasureTask, ExecutionResource>(user, baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(static_cast<Uid>(user));
    }
};

class BaasIdTokenContext
    : public detail::AsyncTask<BaasLoginTaskForIdTokenForApplication, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasIdTokenContext(const user::UserRef& user, const detail::ApplicationInfo& appInfo, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasLoginTaskForIdTokenForApplication, ExecutionResource>(user, appInfo, baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(static_cast<Uid>(user));
    }
};

class BaasProfileUploadContext
    : public detail::AsyncTask<BaasUserProfileUploadTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasProfileUploadContext(const user::UserRef& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasUserProfileUploadTask, ExecutionResource>(user, baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(static_cast<Uid>(user));
    }
};

class BaasProfileSynchronizationContext
    : public detail::AsyncTask<BaasUserProfileSynchronizationTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    BaasProfileSynchronizationContext(const user::UserRef& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<BaasUserProfileSynchronizationTask, ExecutionResource>(user, baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(static_cast<Uid>(user));
    }
};

class DebugNintendoAccountUnlinkageContext
    : public detail::AsyncTask<NintendoAccountUnlinkageTask, ExecutionResource>
{
private:
    const user::UserRef m_User;
public:
    DebugNintendoAccountUnlinkageContext(const user::UserRef& user, BaasOperator& baasOp) NN_NOEXCEPT
        : detail::AsyncTask<NintendoAccountUnlinkageTask, ExecutionResource>(user, baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(static_cast<Uid>(user));
    }
};

/* ------------------------------------------------------------
    BaasAccessTokenAccessorImpl
*/
template <typename Allocator>
inline BaasAccessTokenAccessorImpl<Allocator>::BaasAccessTokenAccessorImpl(
    Allocator& allocator,
    const user::UserRegistry& registry,
    BaasOperator& baasOp,
    Executor& executor) NN_NOEXCEPT
    : m_Allocator(allocator)
    , m_Registry(registry)
    , m_BaasOp(baasOp)
    , m_Executor(executor)
{
}
template <typename Allocator>
inline Result BaasAccessTokenAccessorImpl<Allocator>::EnsureCacheAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    user::UserRef ref;
    NN_RESULT_DO(m_Registry.GetUserRef(&ref, user));
    NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasAccessTokenContext>(
        &m_Allocator, std::move(ref), m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result BaasAccessTokenAccessorImpl<Allocator>::LoadCache(sf::Out<std::uint32_t> pOutActualSize, const sf::OutBuffer& pOut, const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    user::UserRef ref;
    NN_RESULT_DO(m_Registry.GetUserRef(&ref, user));
    NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    size_t sizeActual;
    NN_RESULT_DO(m_BaasOp.LoadAccessTokenCache(&sizeActual, pOut.GetPointerUnsafe(), pOut.GetSize(), user));
    *pOutActualSize = static_cast<uint32_t>(sizeActual);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result BaasAccessTokenAccessorImpl<Allocator>::GetDeviceAccountId(sf::Out<std::uint64_t> pOut, const Uid& user) NN_NOEXCEPT
{
    user::UserRef ref;
    NN_RESULT_DO(m_Registry.GetUserRef(&ref, user));
    NN_SDK_ASSERT(user);
    NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    return m_BaasOp.GetNetworkServiceAccountLoginId(pOut.GetPointer(), user);
}
template <typename Allocator>
inline Result BaasAccessTokenAccessorImpl<Allocator>::RegisterNotificationTokenAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& user, const npns::NotificationToken& nt) NN_NOEXCEPT
{
    bool existence;
    NN_RESULT_DO(m_BaasOp.IsRegistered(&existence, user));
    NN_RESULT_THROW_UNLESS(existence, ResultNetworkServiceAccountRegistrationRequired());

    user::UserRef ref;
    NN_RESULT_DO(m_Registry.GetUserRef(&ref, user));
    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasNotificationRegistrationContext>(
        &m_Allocator, std::move(ref), nt, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result BaasAccessTokenAccessorImpl<Allocator>::UnregisterNotificationTokenAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const Uid& user) NN_NOEXCEPT
{
    bool existence;
    NN_RESULT_DO(m_BaasOp.IsRegistered(&existence, user));
    NN_RESULT_THROW_UNLESS(existence, ResultNetworkServiceAccountRegistrationRequired());

    user::UserRef ref;
    NN_RESULT_DO(m_Registry.GetUserRef(&ref, user));
    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasNotificationErasureContext>(
        &m_Allocator, std::move(ref), m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}

/* ------------------------------------------------------------
    NetworkServiceAccountAdministratorImpl
    */

template <typename Allocator>
inline NetworkServiceAccountAdministratorImpl<Allocator>::NetworkServiceAccountAdministratorImpl(
    Allocator& allocator,
    BaasOperator& baasOp,
    nas::NasOperator& nasOp,
    user::UserRef&& user,
    Executor& executor) NN_NOEXCEPT
    : m_Allocator(allocator)
    , m_BaasOp(baasOp)
    , m_NasOp(nasOp)
    , m_User(std::move(user))
    , m_Executor(executor)
    , m_AppInfo(detail::InvalidApplicationInfo)
{
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::SetApplicationInfo(const detail::ApplicationInfo& appInfo) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(
        false
        || appInfo.mediaType == detail::ApplicationMediaType::GameCard
        || appInfo.mediaType == detail::ApplicationMediaType::Digital,
        ResultInvalidApplication());
    m_AppInfo = appInfo;
    m_BaasOp.InvalidateUserServiceEntryRequirementCache(m_User, m_AppInfo.launchProperty.id);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::SetSystemProgramIdentification(const SystemProgramIdentification& identification, const Bit64& pid) NN_NOEXCEPT
{
    NN_UNUSED(pid);
    detail::ApplicationInfo appInfo = detail::ApplicationInfo::Get(identification.id.value, identification.version, detail::ApplicationMediaType::System);
    NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
    m_AppInfo = appInfo;
    m_BaasOp.InvalidateUserServiceEntryRequirementCache(m_User, m_AppInfo.launchProperty.id);
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::GetServiceEntryRequirementCache(sf::Out<uint32_t> pOut, const ApplicationId& appId) NN_NOEXCEPT
{
    ncm::ApplicationId ncmAppId = {appId.value};
    BaasUserServiceEntryRequirement req;
    NN_RESULT_DO(m_BaasOp.GetUserServiceEntryRequirementCache(&req, m_User, ncmAppId));
    *pOut.GetPointer() = req;
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::InvalidateServiceEntryRequirementCache(const ApplicationId& appId) NN_NOEXCEPT
{
    ncm::ApplicationId ncmAppId = {appId.value};
    m_BaasOp.InvalidateUserServiceEntryRequirementCache(m_User, ncmAppId);
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::InvalidateTokenCache(const ApplicationId& appId) NN_NOEXCEPT
{
    NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));

    ncm::ApplicationId ncmAppId = {appId.value};
    m_BaasOp.InvalidateTokenCache(m_User, ncmAppId);
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::IsRegistered(sf::Out<bool> pOut) NN_NOEXCEPT
{
    return m_BaasOp.IsRegistered(pOut.GetPointer(), m_User);
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::RegisterAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
{
    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasRegistrationContext>(
        &m_Allocator, m_User, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::UnregisterAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
{
    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasErasureContext>(
        &m_Allocator, m_User, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::DeleteRegistrationInfoLocally() NN_NOEXCEPT
{
    return m_BaasOp.DeleteForcibly(m_User);
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CheckAvailability() NN_NOEXCEPT
{
    return static_cast<bool>(m_AppInfo)
        ? m_BaasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(m_User, m_AppInfo.launchProperty.id)
        : m_BaasOp.CheckAvailability(m_User);
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::GetAccountId(sf::Out<account::NetworkServiceAccountId> pOutId) NN_NOEXCEPT
{
    NN_RESULT_DO(static_cast<bool>(m_AppInfo)
        ? m_BaasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(m_User, m_AppInfo.launchProperty.id)
        : m_BaasOp.CheckAvailability(m_User));
    return m_BaasOp.GetNetworkServiceAccountId(pOutId.GetPointer(), m_User);
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::EnsureIdTokenCacheAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());
    NN_RESULT_DO(m_BaasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(m_User, m_AppInfo.launchProperty.id));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasIdTokenContext>(
        &m_Allocator, m_User, m_AppInfo, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::LoadIdTokenCache(sf::Out<std::uint32_t> pOutActualSize, const sf::OutBuffer& pOut) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());
    NN_RESULT_DO(m_BaasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(m_User, m_AppInfo.launchProperty.id));

    size_t sizeActual;
    NN_RESULT_DO(m_BaasOp.LoadIdTokenCache(&sizeActual, pOut.GetPointerUnsafe(), pOut.GetSize(), m_User, m_AppInfo));
    *pOutActualSize.GetPointer() = static_cast<uint32_t>(sizeActual);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::SynchronizeProfileAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
{
    // アプリ非依存の処理
    NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasProfileSynchronizationContext>(
        &m_Allocator, m_User, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::SynchronizeProfileAsyncIfSecondsElapsed(sf::Out<bool> pOutMatched, sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext, uint32_t seconds) NN_NOEXCEPT
{
    // アプリ非依存の処理
    NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));

    TimeSpan sinceLastUpdate;
    auto r = m_BaasOp.GetTimeSpanSinceLastSyncOfUserProfile(&sinceLastUpdate, m_User);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultResourceCacheUnavailable::Includes(r), r);
    if (r.IsSuccess() && sinceLastUpdate < TimeSpan::FromSeconds(seconds))
    {
        *pOutMatched = false;
        NN_RESULT_SUCCESS;
    }
    *pOutMatched = true;
    return SynchronizeProfileAsync(pOutContext);
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::UploadProfileAsync(sf::Out<sf::SharedPointer<account::detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
{
    // アプリ非依存の処理
    NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, BaasProfileUploadContext>(
        &m_Allocator, m_User, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::IsLinkedWithNintendoAccount(sf::Out<bool> pOut) NN_NOEXCEPT
{
    return m_BaasOp.IsLinkedWithNintendoAccount(pOut.GetPointer(), m_User);
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CreateProcedureToLinkWithNintendoAccountImpl(
    sf::EmplacedRef<nas::IOAuthProcedureForNintendoAccountLinkage, nas::NintendoAccountLinkageProcedureImpl<Allocator>>* pOut) NN_NOEXCEPT
{
    // アプリ非依存の処理
    auto r = m_BaasOp.CheckAvailability(m_User);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountLinkageRequired::Includes(r), r);

    bool linkage;
    NN_RESULT_DO(m_BaasOp.IsLinkedWithNintendoAccount(&linkage, m_User));
    NN_RESULT_THROW_UNLESS(!linkage, ResultNintendoAccountAlreadyLinked());

    auto p = Factory::template CreateSharedEmplaced<nas::IOAuthProcedureForNintendoAccountLinkage, nas::NintendoAccountLinkageProcedureImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_Executor, m_User, m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CreateProcedureToLinkWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage>> pOut) NN_NOEXCEPT
{
    sf::EmplacedRef<nas::IOAuthProcedureForNintendoAccountLinkage, nas::NintendoAccountLinkageProcedureImpl<Allocator>> ptr;
    NN_RESULT_DO(CreateProcedureToLinkWithNintendoAccountImpl(&ptr));
    *pOut = sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage>(std::move(ptr));
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::ResumeProcedureToLinkWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    sf::EmplacedRef<nas::IOAuthProcedureForNintendoAccountLinkage, nas::NintendoAccountLinkageProcedureImpl<Allocator>> ptr;
    NN_RESULT_DO(CreateProcedureToLinkWithNintendoAccountImpl(&ptr));
    NN_RESULT_DO(ptr.GetImpl().Resume(sessionId));
    *pOut = sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage>(std::move(ptr));
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CreateProcedureToUpdateLinkageStateOfNintendoAccountImpl(
    sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountCredentialUpdateProcedureImpl<Allocator>>* pOut) NN_NOEXCEPT
{
    // アプリ非依存の処理
    auto r = m_BaasOp.CheckAvailability(m_User);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(r), r);

    auto p = Factory::template CreateSharedEmplaced<http::IOAuthProcedure, nas::NintendoAccountCredentialUpdateProcedureImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_Executor, m_User, m_User, m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CreateProcedureToUpdateLinkageStateOfNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut) NN_NOEXCEPT
{
    sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountCredentialUpdateProcedureImpl<Allocator>> ptr;
    NN_RESULT_DO(CreateProcedureToUpdateLinkageStateOfNintendoAccountImpl(&ptr));
    *pOut = sf::SharedPointer<http::IOAuthProcedure>(std::move(ptr));
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::ResumeProcedureToUpdateLinkageStateOfNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountCredentialUpdateProcedureImpl<Allocator>> ptr;
    NN_RESULT_DO(CreateProcedureToUpdateLinkageStateOfNintendoAccountImpl(&ptr));
    NN_RESULT_DO(ptr.GetImpl().Resume(sessionId));
    *pOut = sf::SharedPointer<http::IOAuthProcedure>(std::move(ptr));
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CreateProcedureToLinkNnidWithNintendoAccountImpl(
    sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountNnidLinkageProcedureImpl<Allocator>>* pOut) NN_NOEXCEPT
{
    // アプリ非依存の処理
    NN_RESULT_DO(m_NasOp.CheckAvailability(m_User));

    auto p = Factory::template CreateSharedEmplaced<http::IOAuthProcedure, nas::NintendoAccountNnidLinkageProcedureImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_Executor, m_User, m_User, m_NasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::CreateProcedureToLinkNnidWithNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut) NN_NOEXCEPT
{
    sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountNnidLinkageProcedureImpl<Allocator>> ptr;
    NN_RESULT_DO(CreateProcedureToLinkNnidWithNintendoAccountImpl(&ptr));
    *pOut = sf::SharedPointer<http::IOAuthProcedure>(std::move(ptr));
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::ResumeProcedureToLinkNnidWithNintendoAccount(sf::Out<sf::SharedPointer<http::IOAuthProcedure>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    sf::EmplacedRef<http::IOAuthProcedure, nas::NintendoAccountNnidLinkageProcedureImpl<Allocator>> ptr;
    NN_RESULT_DO(CreateProcedureToLinkNnidWithNintendoAccountImpl(&ptr));
    NN_RESULT_DO(ptr.GetImpl().Resume(sessionId));
    *pOut = sf::SharedPointer<http::IOAuthProcedure>(std::move(ptr));
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::DebugUnlinkNintendoAccountAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT
{
    // アプリ非依存の処理
    auto r = m_BaasOp.CheckAvailability(m_User);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(r), r);

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, DebugNintendoAccountUnlinkageContext>(
        &m_Allocator, m_User, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}
template <typename Allocator>
inline Result NetworkServiceAccountAdministratorImpl<Allocator>::DebugSetAvailabilityErrorDetail(uint32_t rawError) NN_NOEXCEPT
{
    switch (rawError)
    {
    // Network Service Account itself
    case detail::NsaAvailabilityError_NetworkServiceAccountCredentialBroken:
        return m_BaasOp.DegradeNetworkServiceAccountUserState(m_User, BaasUserState_Inconsequent);
    case detail::NsaAvailabilityError_NetworkServiceAccountUnmanaged:
        return m_BaasOp.DegradeNetworkServiceAccountUserState(m_User, BaasUserState_Unmanaged);
    case detail::NsaAvailabilityError_NetworkServiceAccountBanned:
        return m_BaasOp.DegradeNetworkServiceAccountUserState(m_User, BaasUserState_Banned);

    // Linkage State
    case detail::NsaAvailabilityError_NintendoAccountLinkageBroken:
        return m_BaasOp.DegradeNintendoAccountLinkageState(m_User);

    // Linked User Resource
    case detail::NsaAvailabilityError_NintendoAccountStateOtherButInteractionRequired:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_OtherButInteractionRequired);
    case detail::NsaAvailabilityError_NintendoAccountStateDeleted:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_Deleted);
    case detail::NsaAvailabilityError_NintendoAccountStateBanned:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_Banned);
    case detail::NsaAvailabilityError_NintendoAccountStateSuspended:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_Suspended);
    case detail::NsaAvailabilityError_NintendoAccountStateWithdrawn:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_Withdrawn);
    case detail::NsaAvailabilityError_NintendoAccountStateTermsAgreementRequired:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_TermsAgreementRequired);
    case detail::NsaAvailabilityError_NintendoAccountStateReauthorizationRequired:
        return m_BaasOp.DegradeLinkedNintendoAccountUserState(m_User, NintendoAccountUserState_ReauthorizationRequired);

    default:
        NN_RESULT_THROW(ResultNotSupported());
    }
}

}}} // ~namespace nn::account::baas
