﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/account/detail/account_CacheBase.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_Types.h>

#include <nn/nn_TimeSpan.h>

namespace nn { namespace account { namespace baas {

class BaasUserServiceEntryRequirementCache
    : detail::CacheBaseN<16, std::pair<Uid, ncm::ApplicationId>, BaasUserServiceEntryRequirement, detail::DefaultCacheOperator<BaasUserServiceEntryRequirement>, detail::DefaultCacheResult>
{
private:
    typedef std::pair<Uid, ncm::ApplicationId> Tag;
    typedef detail::CacheBaseN<16, Tag, BaasUserServiceEntryRequirement, detail::DefaultCacheOperator<BaasUserServiceEntryRequirement>, detail::DefaultCacheResult> Base;

public:
    Result Initialize(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        return Base::Initialize(storage);
    }
    void Store(const Uid& uid, const ncm::ApplicationId& appId, const BaasUserServiceEntryRequirement& requirement) NN_NOEXCEPT
    {
        Tag tag(uid, appId);
        Base::Store(tag, 0, requirement);
    }
    Result Get(BaasUserServiceEntryRequirement* pOut, const Uid& uid, const ncm::ApplicationId& appId) const NN_NOEXCEPT
    {
        std::lock_guard<const Base> lock(*static_cast<const Base*>(this));
        Tag tag(uid, appId);
        return Base::FindUnsafe(pOut, tag);
    }
    void Invalidate(const Uid& uid) NN_NOEXCEPT
    {
        Base::InvalidateIfMatch([&](const Tag& tag) -> bool {
            return tag.first == uid;
        });
    }
    void Invalidate(const Uid& uid, const ncm::ApplicationId& appId) NN_NOEXCEPT
    {
        Tag tag(uid, appId);
        Base::Invalidate(tag);
    }
    Result CheckRequirement(const Uid& uid, const ncm::ApplicationId& appId) const NN_NOEXCEPT
    {
        BaasUserServiceEntryRequirement requirement;
        NN_RESULT_DO(Get(&requirement, uid, appId));
        switch (requirement)
        {
        case BaasUserServiceEntryRequirement_None:
            NN_RESULT_SUCCESS;
        case BaasUserServiceEntryRequirement_Op2CommonLicense:
            NN_RESULT_THROW(ResultOp2CommonLicenseRequired());
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

}}} // ~namespace nn::account::baas
