﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/detail/account_Notifiable.h>
#include <nn/account/http/account_OAuthTypes.h>
#include <nn/account/nas/account_NasUserResourceCache.h>
#include <nn/account/nas/account_NasInternalTypes.h>
#include <nn/account/user/account_UserRegistry.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_Types.h>

#include <mutex>

#include <nn/nn_Result.h>
#include <nn/os/os_ReaderWriterLock.h>

namespace nn {
namespace account {
namespace baas {

enum BaasUserInfoHolderTag
{
    BaasUserInfoHolderTag_AvailablityChange,
};

class BaasUserInfoHolder
    : detail::Notifiable<BaasUserInfoHolderTag, detail::BaasUserInfoHolderNotifierCountMax>
{
private:
    mutable os::ReaderWriterLock m_RwLock;

    nas::NasUserResourceCache* m_pNasUserCache;
    const detail::AbstractLocalStorage* m_pStorage;

    struct BaasUserData
    {
        BaasUserInfo info;
        BaasCredential credential;
    };
    mutable struct Cache
    {
        struct Entry
        {
            Uid user;
            BaasUserData data;

            void Invalidate() NN_NOEXCEPT
            {
                user = InvalidUid;
            }
                NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
            {
                return user;
            }
        } entries[UserCountMax * 2];

        static const Entry InvalidEntry;

        Cache() NN_NOEXCEPT;
        const Entry& Find(const Uid& user) const NN_NOEXCEPT;
        void Update(const Uid& user, const BaasUserData& data) NN_NOEXCEPT;
        void Remove(const Uid& user) NN_NOEXCEPT;
    } m_Cache;

    Result StoreImplUnsafe(const Uid& user, const BaasUserInfo& userInfo, const BaasCredential& credential) NN_NOEXCEPT;
    Result LoadImplUnsafe(BaasUserInfo* pOutUserInfo, BaasCredential* pOutCredential, const Uid& user) const NN_NOEXCEPT;
    Result LoadImplUnsafe(BaasUserInfo* pOutUserInfo, const Uid& user) const NN_NOEXCEPT;
    Result IsRegisteredUnsafe(bool* pOut, const Uid& user) const NN_NOEXCEPT;

public:
    BaasUserInfoHolder() NN_NOEXCEPT;
    Result Initialize(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    Result Initialize(nas::NasUserResourceCache& nasUserCache, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;

    Result Register(const Uid& user, const NetworkServiceAccountId& id, const BaasCredential& credential) NN_NOEXCEPT;
    Result Unregister(const Uid& user) NN_NOEXCEPT;
    Result IsRegistered(bool* pOut, const Uid& user) const NN_NOEXCEPT;
    Result IsLinkedWithNintendoAccount(bool* pOut, const Uid& user) const NN_NOEXCEPT;

    // BaasUserInfoHolderNotifier からの呼び出し用
    void ReleaseSystemEvent(const os::SystemEventType *pEvent) NN_NOEXCEPT
    {
        Notifiable::ReleaseSystemEvent(pEvent);
    }

    // 通知オブジェクト提供
    Result GetBaasUserAvailabilityChangeNotifier(os::SystemEventType** pOutEvent) NN_NOEXCEPT
    {
        return Notifiable::AcquireSystemEvent(pOutEvent, BaasUserInfoHolderTag_AvailablityChange);
    }

    // BaaS User の状態が正常であれば参照してよい
    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOut, const Uid& user) const NN_NOEXCEPT;
    Result GetLinkedNintendoAccountId(NintendoAccountId* pOut, const Uid& user) const NN_NOEXCEPT;
    Result GetCredential(BaasCredential* pOut, const Uid& user) const NN_NOEXCEPT;

    // Network Service Account の利用ポリシーを満たしていれば参照してよい
    Result CheckAvailability(const Uid& user) const NN_NOEXCEPT;

    // 内部の連携処理等で使用
    // - Network Service Account
    Result UpdateNetworkServiceAccountId(const Uid& user, const NetworkServiceAccountId& nsaId) NN_NOEXCEPT;
    Result DegradeBaasUserState(const Uid& user, BaasUserState state) NN_NOEXCEPT;

    // - Nintendo Account 連携状態
    Result InitializeNintendoAccountLinkageState(
        const Uid& user, const NintendoAccountId& naId,
        const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
        void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;
    Result CompleteNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT;
    Result DegradeNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT;
    Result DismissNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT;

    // - Nintendo Account ユーザー状態
    Result DegradeNintendoAccountUserState(const Uid& user, NintendoAccountUserState state) NN_NOEXCEPT;
    Result RecoveryNintendoAccountUserState(
        const Uid& user, const NintendoAccountId& naId,
        const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
        void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;
    Result RecoveryNintendoAccountUserState(const Uid& user) NN_NOEXCEPT;

    void lock() const NN_NOEXCEPT;
    void unlock() const NN_NOEXCEPT;
    void lock_shared() const NN_NOEXCEPT;
    void unlock_shared() const NN_NOEXCEPT;
};

} // ~namespace nn::account::baas
}
}
