﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasChannelDriver.h>
#include <nn/account/baas/account_BaasLoginDriver.h>
#include <nn/account/baas/account_BaasRegistrationDriver.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/baas/account_BaasUserDriver.h>
#include <nn/account/baas/account_BaasUserInfoHolder.h>
#include <nn/account/baas/account_BaasUserResourceCache.h>
#include <nn/account/baas/account_BaasUserServiceEntryRequirementCache.h>
#include <nn/account/baas/account_ResultForBaas.h>
#include <nn/account/detail/account_Log.h>
#include <nn/account/http/account_OAuthTypes.h>
#include <nn/account/account_RuntimeResource.h>

#include <nn/nn_Result.h>

namespace nn { namespace account {

namespace profile {
class ProfileAdaptor;
struct ProfileBase;
class ProfileStorage;
} // ~namespace nn::account::profile

namespace nas {
struct NasCredentialCache;
} // ~namespace nn::account::nas
}} // ~namespace nn::account

namespace nn { namespace account { namespace baas {

class BaasOperator
{
private:
    const BaasLoginDriver m_BaasLoginDriver;
    const BaasRegistrationDriver m_BaasRegistrationDriver;
    const BaasUserDriver m_BaasUserDriver;
    const BaasChannelDriver m_BaasChannelDriver;
    BaasUserInfoHolder& m_UserInfoHolder;
    profile::ProfileStorage& m_ProfileStorage;
    ndas::NdasOperator& m_NdasOperator;
    const detail::AbstractLocalStorage& m_Storage;

    UserAccessTokenCache& m_UserAccessTokenCache;
    UserIdTokenCache& m_UserIdTokenCache;
    BaasUserResourceCache m_BaasUserResourceCache;
    BaasUserServiceEntryRequirementCache m_BaasUserServiceEntryRequirementCache;

    Result GetUserInfo(
        NetworkServiceAccountId* pOutId, BaasCredential* pOutCredential, util::optional<NintendoAccountId> *ppOutNaId,
        const Uid& user) NN_NOEXCEPT;
    Result HandleUserLoginResult(const Result result, const Uid& user) NN_NOEXCEPT;
    Result HandleUserLoginResult(const Result result, const Uid& user, const ncm::ApplicationId& appId) NN_NOEXCEPT;
    Result EnsureUserAccessTokenCacheImpl(
        NetworkServiceAccountId* pOutId, BaasCredential* pOutCredential, util::optional<NintendoAccountId>* ppNaId,
        const Uid& user,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result EnsureUserIdTokenCacheImpl(
        NetworkServiceAccountId* pOutId, BaasCredential* pOutCredential, util::optional<NintendoAccountId>* ppNaId,
        const Uid& user, const detail::ApplicationInfo& appInfo,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

public:
    BaasOperator(
        ClientAccessTokenCache& clientAccessTokenCache,
        UserAccessTokenCache& userAccessTokenCache,
        UserIdTokenCache& userIdTokenCache,
        BaasUserInfoHolder& userInfoHolder,
        profile::ProfileStorage& profile,
        ndas::NdasOperator& ndasOperator,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;

    Result GetBaasUserAvailabilityChangeNotifier(os::SystemEventType** ppOut) NN_NOEXCEPT
    {
        return m_UserInfoHolder.GetBaasUserAvailabilityChangeNotifier(ppOut);
    }
    void ReleaseBaasUserAvailabilityChangeNotifier(const os::SystemEventType* pEvent) NN_NOEXCEPT
    {
        m_UserInfoHolder.ReleaseSystemEvent(pEvent);
    }

    Result Register(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result Import(
        const Uid& user,
        const NetworkServiceAccountId& id, const BaasCredential& credential,
        const NintendoAccountId& naId, const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result Unregister(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result DeleteForcibly(const Uid& user) NN_NOEXCEPT;
    Result IsRegistered(bool* pOut, const Uid& user) const NN_NOEXCEPT;
    Result CheckAvailability(const Uid& user) const NN_NOEXCEPT;
    Result CheckAvailabilityWithUserServiceEntryRequirementCache(const Uid& user, const ncm::ApplicationId& appId) const NN_NOEXCEPT;
    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOut, const Uid& user) const NN_NOEXCEPT;
    Result GetNetworkServiceAccountLoginId(uint64_t* pOut, const Uid& user) const NN_NOEXCEPT;

    Result LoadAccessTokenCache(
        size_t* pOutSizeActual, char* buffer, size_t bufferSize,
        const Uid& user) const NN_NOEXCEPT;
    Result LoadIdTokenCache(
        size_t* pOutSizeActual, char* buffer, size_t bufferSize,
        const Uid& user, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT;
    Result LoadIdTokenCache(
        size_t* pOutSizeActual, char* buffer, size_t bufferSize,
        const NetworkServiceAccountId& nsaId, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT;
    void InvalidateTokenCache(const Uid& uid) NN_NOEXCEPT;
    void InvalidateTokenCache(const Uid& uid, const ncm::ApplicationId& appId) NN_NOEXCEPT;

    Result EnsureAccessTokenCache(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result EnsureIdTokenCacheForApplication(const Uid& user, const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result EnsureIdTokenCacheForApplication(
        const NetworkServiceAccountId& nsaId, const BaasCredential& credential, const NintendoAccountId& naId, const detail::ApplicationInfo& appInfo,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result UploadUserProfile(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result SynchronizeUserProfile(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result GetTimeSpanSinceLastSyncOfUserProfile(TimeSpan* pOut, const Uid& user) NN_NOEXCEPT;
    void InvalidateUserResourceCache(const Uid& user) NN_NOEXCEPT;

    Result RegisterNotificationToken(const Uid& user, const npns::NotificationToken& nt, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result DownloadNotificationToken(npns::NotificationToken* pOut, const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result UnregisterNotificationToken(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result LinkWithNintendoAccount(
        bool* pOutIsNetworkServiceAccountReplaced,
        const Uid& user,
        const NintendoAccountId& naId, const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result UpdateLinkageWithNintendoAccount(
        const Uid& user,
        const NintendoAccountId& naId, const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result UnlinkNintendoAccount(
        const Uid& user,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result IsLinkedWithNintendoAccount(bool* pOut, const Uid& user) const NN_NOEXCEPT;
    Result GetLinkedNintendoAccountId(NintendoAccountId* pOut, const Uid& user) const NN_NOEXCEPT;

    Result AcquireCredentialFromNintendoAccountIdToken(
        NetworkServiceAccountId* pOutNsaId, BaasCredential* pOutCredential, UserProfile* pOutProfile,
        const NintendoAccountId& naId, const detail::Uuid& nasIdTokenCacheId,
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result EnsureIdTokenCacheForApplicationForGuest(
        GuestUserProfile* pOut,
        const detail::ApplicationInfo& appInfo, const char (&nasIdToken)[detail::NasIdTokenSizeMax],
        const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

    Result GetUserServiceEntryRequirementCache(BaasUserServiceEntryRequirement* pOut, const Uid& user, const ncm::ApplicationId& appId) const NN_NOEXCEPT;
    void InvalidateUserServiceEntryRequirementCache(const Uid& user, const ncm::ApplicationId& appId) NN_NOEXCEPT;
    void InvalidateUserServiceEntryRequirementCache(const Uid& user) NN_NOEXCEPT;
    void SetUserServiceEntryRequirementCache(const Uid& user, const ncm::ApplicationId& appId, BaasUserServiceEntryRequirement requirement) NN_NOEXCEPT;

    Result DegradeNetworkServiceAccountUserState(const Uid& user, BaasUserState state) NN_NOEXCEPT;
    Result DegradeNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT;
    Result DegradeLinkedNintendoAccountUserState(const Uid& user, NintendoAccountUserState state) NN_NOEXCEPT;
    Result RecoveryLinkedNintendoAccountUserState(const Uid& user) NN_NOEXCEPT;
};

class BaasRegistrationTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_BaasOp.Register(m_User, resource, this);
    }
public:
    BaasRegistrationTask(const Uid& user, BaasOperator& baasOp) NN_NOEXCEPT
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class BaasErasureTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        auto r = m_BaasOp.Unregister(m_User, resource, this);
        NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r) || ResultBaasStatus::Includes(r), r);
        if (!r.IsSuccess())
        {
            return m_BaasOp.DeleteForcibly(m_User);
        }
        NN_RESULT_SUCCESS;
    }
public:
    BaasErasureTask(const Uid& user, BaasOperator& baasOp) NN_NOEXCEPT
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class BaasLoginTaskForAccessToken
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));
        return m_BaasOp.EnsureAccessTokenCache(m_User, resource, this);
    }
public:
    BaasLoginTaskForAccessToken(const Uid& user, BaasOperator& baasOp) NN_NOEXCEPT
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class BaasLoginTaskForIdTokenForApplication
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
    const detail::ApplicationInfo m_AppInfo;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));
        NN_RESULT_DO(m_BaasOp.EnsureIdTokenCacheForApplication(m_User, m_AppInfo, resource, this));
        NN_RESULT_SUCCESS;
    }
public:
    BaasLoginTaskForIdTokenForApplication(const Uid& user, const detail::ApplicationInfo& appInfo, BaasOperator& baasOp) NN_NOEXCEPT
        : m_BaasOp(baasOp)
        , m_User(user)
        , m_AppInfo(appInfo)
    {
        NN_SDK_ASSERT(m_User);
        NN_SDK_ASSERT(m_AppInfo);
    }
};

class BaasUserProfileUploadTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));
        return m_BaasOp.UploadUserProfile(m_User, resource, this);
    }
public:
    BaasUserProfileUploadTask(const Uid& user, BaasOperator& baasOp)
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class BaasUserProfileSynchronizationTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_RESULT_DO(m_BaasOp.CheckAvailability(m_User));
        return m_BaasOp.SynchronizeUserProfile(m_User, resource, this);
    }
public:
    BaasUserProfileSynchronizationTask(const Uid& user, BaasOperator& baasOp)
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class BaasNotificationRegistrationTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
    const npns::NotificationToken m_Nt;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_BaasOp.RegisterNotificationToken(m_User, m_Nt, resource, this);
    }
public:
    BaasNotificationRegistrationTask(const Uid& user, const npns::NotificationToken& nt, BaasOperator& baasOp)
        : m_BaasOp(baasOp)
        , m_User(user)
        , m_Nt(nt)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class BaasNotificationErasureTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_BaasOp.UnregisterNotificationToken(m_User, resource, this);
    }
public:
    BaasNotificationErasureTask(const Uid& user, BaasOperator& baasOp)
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class DebugBaasNotificationDownloadTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
    npns::NotificationToken* const m_pOutNt;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_BaasOp.DownloadNotificationToken(m_pOutNt, m_User, resource, this);
    }
public:
    DebugBaasNotificationDownloadTask(npns::NotificationToken* pOutNt, const Uid& user, BaasOperator& baasOp)
        : m_BaasOp(baasOp)
        , m_User(user)
        , m_pOutNt(pOutNt)
    {
        NN_SDK_ASSERT(m_User);
    }
};

class NintendoAccountUnlinkageTask
    : public detail::Executable<ExecutionResource>
{
private:
    BaasOperator& m_BaasOp;
    const Uid m_User;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        auto r = m_BaasOp.CheckAvailability(m_User);
        NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(r), r);
        return m_BaasOp.UnlinkNintendoAccount(m_User, resource, this);
    }
public:
    NintendoAccountUnlinkageTask(const Uid& user, BaasOperator& baasOp)
        : m_BaasOp(baasOp)
        , m_User(user)
    {
        NN_SDK_ASSERT(m_User);
    }
};

}}} // ~namespace account::baas
