﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_Execution.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/os/os_MultipleWaitTypes.h>

namespace nn { namespace account {

template <typename ExecutorType, typename ExecutionResourceType>
class ExecutionManager
{
private:
    ExecutorType& m_Executor;
    os::MultiWaitHolderType m_ExecutorHolder;
    os::Event m_Terminator;
    os::MultiWaitHolderType m_TerminatorHolder;

    bool m_IsLinked;

public:
    explicit ExecutionManager(ExecutorType& executor) NN_NOEXCEPT
        : m_Executor(executor)
        , m_Terminator(os::EventClearMode_ManualClear)
        , m_IsLinked(false)
    {
        m_Executor.InitializeMultiWaitHolder(&m_ExecutorHolder);
        os::SetMultiWaitHolderUserData(&m_ExecutorHolder, reinterpret_cast<uintptr_t>(this));
        os::InitializeMultiWaitHolder(&m_TerminatorHolder, m_Terminator.GetBase());
        os::SetMultiWaitHolderUserData(&m_TerminatorHolder, reinterpret_cast<uintptr_t>(this));
    }
    ~ExecutionManager() NN_NOEXCEPT
    {
        os::FinalizeMultiWaitHolder(&m_ExecutorHolder);
        os::FinalizeMultiWaitHolder(&m_TerminatorHolder);
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_IsLinked;
    }
    void Link(os::MultiWaitType* pWaiter) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsLinked);
        os::LinkMultiWaitHolder(pWaiter, &m_ExecutorHolder);
        os::LinkMultiWaitHolder(pWaiter, &m_TerminatorHolder);
        m_IsLinked = true;
    }
    void Unlink() NN_NOEXCEPT
    {
        if (m_IsLinked)
        {
            os::UnlinkMultiWaitHolder(&m_ExecutorHolder);
            os::UnlinkMultiWaitHolder(&m_TerminatorHolder);
            m_IsLinked = false;
        }
    }
    bool Execute(
        const os::MultiWaitHolderType* pSignaled,
        const ExecutionResourceType& resource) NN_NOEXCEPT
    {
        if (pSignaled == &m_TerminatorHolder)
        {
            m_Terminator.Clear();
            return false;
        }
        else if (pSignaled == &m_ExecutorHolder)
        {
            m_Executor.ClearSignal();
            m_Executor.Execute(resource);
            return true;
        }
        NN_ABORT("[nn::account] Enexpected execution signaled (internal)\n");
    }
    void RequestStop() NN_NOEXCEPT
    {
        m_Terminator.Signal();
    }
};

}} // ~namespace nn::account
