﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <curl/curl.h>

#include <nn/account/detail/account_Execution.h>
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace account {
class LargeBufferAllocator;
}} // ~namespace nn::account

namespace nn { namespace account {

const size_t RequiredBufferSize = 16 * 1024;
const size_t RequiredLargeBufferSize = 512 * 1024;

struct ExecutionResource
{
    CURL* curlHandle;
    struct
    {
        void* address;
        size_t size;
    } buffer;
    LargeBufferAllocator* pLargeBufferAllocator;
};

class Executor : public detail::Executor<detail::QueueableTaskCountMax, ExecutionResource> {};

class LargeBuffer
{
    friend class LargeBufferAllocator;

private:
    os::SdkMutex& m_Lock;
    void* m_Address;
    size_t m_Size;

    LargeBuffer(void* address, size_t size, os::SdkMutex& lock) NN_NOEXCEPT;

public:
    ~LargeBuffer() NN_NOEXCEPT;
    void* GetAddress() const NN_NOEXCEPT;
    size_t GetSize() const NN_NOEXCEPT;
};

class LargeBufferAllocator
{
private:
    os::SdkMutex m_Lock;
    void* m_Address;
    size_t m_Size;

public:
    LargeBufferAllocator(void* address, size_t size);
    LargeBuffer Allocate() NN_NOEXCEPT;
};

}} // ~namespace nn::account

/* --------------------------------------------------------------------------------------------
    実装
 */
namespace nn { namespace account {

inline LargeBuffer::LargeBuffer(void* address, size_t size, os::SdkMutex& lock) NN_NOEXCEPT
    : m_Lock(lock)
    , m_Address(address)
    , m_Size(size)
{
    NN_SDK_ASSERT(address != nullptr);
    NN_SDK_ASSERT(size >= RequiredLargeBufferSize);
    NN_SDK_ASSERT(lock.IsLockedByCurrentThread());
}
inline LargeBuffer::~LargeBuffer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    m_Lock.Unlock();
}
inline void* LargeBuffer::GetAddress() const NN_NOEXCEPT
{
    return m_Address;
}
inline size_t LargeBuffer::GetSize() const NN_NOEXCEPT
{
    return m_Size;
}

inline LargeBufferAllocator::LargeBufferAllocator(void* address, size_t size)
    : m_Address(address)
    , m_Size(size)
{
    NN_SDK_ASSERT(address != nullptr);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(address) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_ASSERT(size >= RequiredLargeBufferSize);
}
inline LargeBuffer LargeBufferAllocator::Allocate() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Lock.IsLockedByCurrentThread());
    m_Lock.Lock();
    return LargeBuffer(m_Address, m_Size, m_Lock);
}

}} // ~namespace nn::account
