﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_AccountServiceImpl.h>

#include <nn/account/baas/account_NetworkServiceAccountGuestLoginImpl.h>
#include <nn/account/baas/account_NetworkServiceAccountFloatingRegistrationImpl.h>
#include <nn/account/baas/account_NotifierImpl.h>
#include <nn/account/detail/account_Log.h>
#include <nn/account/detail/account_NotifierImpl.h>
#include <nn/account/detail/account_ReferenceContext.h>
#include <nn/account/detail/account_RightsUtil.h>
#include <nn/account/detail/account_Settings.h>
#include <nn/account/profile/account_ProfileEditorImpl.h>
#include <nn/account/user/account_UserRegistry.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_ResultForAdministrators.h>

#include <limits>

#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/arp/arp_Api.h>
#include <nn/arp/arp_Result.h>
#include <nn/fs/fs_SaveDataThumbnail.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/srepo/srepo_StateNotifier.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nn/ovln/ovln_SenderForOverlay.h>
#endif

namespace nn { namespace account {
namespace detail
{
/* UserStateManagerActivator ---------------------------------------------------------------------- */

inline UserStateManagerActivator::UserStateManagerActivator(user::UserStateManager& userStateManager) NN_NOEXCEPT
    : m_UserStateManager(userStateManager)
{
}
inline UserStateManagerActivator::~UserStateManagerActivator() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        m_UserStateManager.Disable(m_ApplicationId);
    }
}
inline Result UserStateManagerActivator::Initialize(ApplicationId appId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_UserStateManager.TryEnable(appId),
        ResultInvalidProtocolAccess());
    m_Initialized = true;
    m_ApplicationId = appId;
    NN_RESULT_SUCCESS;
}

/* ---------------------------------------------------------------------- */

struct NetworkServiceAvailabilityCheckTask
    : public detail::Executable<ExecutionResource>
{
private:
    baas::BaasOperator& m_BaasOp;
    const detail::ApplicationInfo m_AppInfo;
    Uid m_Users[UserCountMax];

protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        bool licenseRequired = false;
        for (auto i = 0; i < std::extent<decltype(m_Users)>::value && m_Users[i]; ++ i)
        {
            if (this->IsCancelled())
            {
                NN_RESULT_THROW(ResultCancelled());
            }

            auto r = m_BaasOp.EnsureIdTokenCacheForApplication(m_Users[i], m_AppInfo, resource, this);
            NN_RESULT_THROW_UNLESS(ResultNetworkServiceAccountUnavailable::Includes(r), r);
            licenseRequired = (licenseRequired || ResultLicenseRequired::Includes(r));
        }
        if (licenseRequired)
        {
            NN_RESULT_THROW(ResultNoLicensedNetworkServiceAccounts());
        }
        NN_RESULT_THROW(ResultNoValidNetworkServiceAccounts());
    }
public:
    NetworkServiceAvailabilityCheckTask(const detail::ApplicationInfo& appInfo, const Uid (&users)[UserCountMax], baas::BaasOperator& baasOp) NN_NOEXCEPT
        : m_BaasOp(baasOp)
        , m_AppInfo(appInfo)
    {
        for (auto i = 0; i < std::extent<decltype(m_Users)>::value; ++ i)
        {
            m_Users[i] = users[i];
        }
    }
};

template <typename Allocator>
class OpenContextRetentionActivator
    : UserStateManagerActivator
{
    OpenContextRetainer<Allocator>& m_OpenContextRetainer;
    bool m_Initialized {false};
    ApplicationId m_ApplicationId;

public:
    OpenContextRetentionActivator(
        user::UserStateManager& userStateManager,
        OpenContextRetainer<Allocator>& openContextRetainer) NN_NOEXCEPT
        : UserStateManagerActivator(userStateManager)
        , m_OpenContextRetainer(openContextRetainer)
    {
    }
    ~OpenContextRetentionActivator() NN_NOEXCEPT
    {
        if (m_Initialized)
        {
            m_OpenContextRetainer.Disable(m_ApplicationId);
        }
    }
    Result Initialize(ApplicationId appId) NN_NOEXCEPT
    {
        NN_RESULT_DO(UserStateManagerActivator::Initialize(appId));
        NN_RESULT_THROW_UNLESS(
            m_OpenContextRetainer.TryEnable(appId),
            ResultInvalidProtocolAccess());
        m_Initialized = true;
        m_ApplicationId = appId;
        NN_RESULT_SUCCESS;
    }
    Result Dummy() const NN_NOEXCEPT
    {
        NN_RESULT_THROW(ResultInvalidProtocolAccess());
    }
};

template <typename Allocator>
class OpenContextRetentionActivatorForDebug
    : public OpenContextRetentionActivator<Allocator>
{
private:
    OpenContextRetainer<Allocator>& m_OpenContextRetainer;

public:
    explicit OpenContextRetentionActivatorForDebug(
        user::UserStateManager& userStateManager,
        OpenContextRetainer<Allocator>& openContextRetainer) NN_NOEXCEPT
        : OpenContextRetentionActivator<Allocator>(userStateManager, openContextRetainer)
        , m_OpenContextRetainer(openContextRetainer)
    {
    }
    Result Initialize() NN_NOEXCEPT
    {
        ApplicationId appIds[ApplicationCountMax];
        auto count = m_OpenContextRetainer.ListApplications(appIds, std::extent<decltype(appIds)>::value);
        if (!(count > 0))
        {
            // 何もしないで成功
            NN_DETAIL_ACCOUNT_WARN("OpenContextRetentionActivatorForDebug: No applications running\n");
            NN_RESULT_SUCCESS;
        }
        return OpenContextRetentionActivator<Allocator>::Initialize(appIds[0]);
    }
};

typedef detail::AsyncTask<NetworkServiceAvailabilityCheckTask, ExecutionResource> NetworkServiceAvailabilityCheckContext;
} // ~namespace nn::account::detail

namespace ndas
{
typedef detail::AsyncTask<ndas::DeviceReAuthenticationTask, ExecutionResource> DeviceReAuthenticationContext;
typedef detail::AsyncTask<ndas::ApplicationReAuthenticationTask, ExecutionResource> ApplicationReAuthenticationContext;
} // ~namespace nn::account::ndas


/* AccountServiceImplBase ---------------------------------------------------------------------- */

#define NN_ACCOUNT_DEFINE_SERVICE_METHOD(rtype, methodInfo) \
template <typename Allocator> \
inline rtype AccountServiceImpl<Allocator>::methodInfo NN_NOEXCEPT

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetBaasAccountManagerForApplication(sf::Out<sf::SharedPointer<baas::IManagerForApplication>>, const Uid&, const Bit64&))
{
    NN_RESULT_THROW(ResultNotSupported());
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetBaasAccountManagerForSystemService(sf::Out<sf::SharedPointer<baas::IManagerForSystemService>> pOutManager, const Uid& uid))
{
    user::UserRef user;
    NN_RESULT_DO(m_UserRegistry.GetUserRef(&user, uid));
    auto p = Factory::template CreateSharedEmplaced<baas::IManagerForSystemService, detail::ReferenceContextImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_NasSessionPool, m_BaasOp, m_NasOp, std::move(user), m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutManager = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetBaasAccountAdministrator(sf::Out<sf::SharedPointer<baas::IAdministrator>> pOutAdministrator, const Uid& uid))
{
    user::UserRef user;
    NN_RESULT_DO(m_UserRegistry.GetUserRef(&user, uid));
    auto p = Factory::template CreateSharedEmplaced<baas::IAdministrator, detail::ReferenceContextImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_NasSessionPool, m_BaasOp, m_NasOp, std::move(user), m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutAdministrator = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetUserCount(sf::Out<std::int32_t> pOutCount))
{
    *pOutCount = static_cast<int32_t>(m_UserRegistry.GetUserCount());
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetUserExistence(sf::Out<bool> pOutExistence, const Uid& uid))
{
    return m_UserRegistry.GetUserExistence(pOutExistence.GetPointer(), uid);
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ListAllUsers(const sf::OutArray<Uid>& outUsers))
{
    NN_SDK_ASSERT(outUsers.GetLength() <= static_cast<uint64_t>(std::numeric_limits<int>::max()));
    return m_UserRegistry.ListAllUsers(outUsers.GetData(), static_cast<int>(outUsers.GetLength()));
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ListOpenUsers(const sf::OutArray<Uid>& outUsers))
{
    NN_RESULT_THROW_UNLESS(outUsers.GetLength() == UserCountMax, ResultInvalidArrayLength());

    Uid users[UserCountMax];
    NN_RESULT_DO(m_UserRegistry.ListAllUsers(users, std::extent<decltype(users)>::value));

    Uid opens[UserCountMax];
    auto openCount = m_UserStateManager.ListOpenUsers(opens, std::extent<decltype(opens)>::value);

    auto contains = [](const Uid list[], int length, const Uid& uid) -> bool {
        return std::find(list, list + length, uid) != (list + length);
    };

    int ct = 0;
    for (int i = 0; i < std::extent<decltype(users)>::value && users[i]; ++i)
    {
        if (contains(opens, openCount, users[i]))
        {
            outUsers.GetData()[ct++] = users[i];
        }
    }
    for (int i = ct; i < static_cast<int>(outUsers.GetLength()); ++i)
    {
        outUsers.GetData()[i] = InvalidUid;
    }
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetLastOpenedUser(sf::Out<Uid> pOutUid))
{
    if (!m_UserStateManager.TryGetLastOpenedUser(pOutUid.GetPointer()))
    {
        *pOutUid = InvalidUid;
    }
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    BeginUserRegistration(sf::Out<Uid> pOutUid))
{
    auto lock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserRegistry.BeginUserRegistration(pOutUid.GetPointer()));
    m_ProfileStorage.Add(*pOutUid); // プロフィールは Pending 状態でも使用可能というポリシー
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CompleteUserRegistration(const Uid& uid))
{
    NN_RESULT_THROW_UNLESS(m_Usage.applicationCount.Get() == 0, ResultRegistrationNotPermitted());
    return CompleteUserRegistrationForcibly(uid);
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CompleteUserRegistrationForcibly(const Uid& uid))
{
    auto lock = m_Storage.AcquireWriterLock();
    if (m_Usage.applicationCount.Get() != 0)
    {
        NN_DETAIL_ACCOUNT_INFO("User account created while application count = %d\n", m_Usage.applicationCount.Get());
    }
    NN_RESULT_DO(m_UserRegistry.CompleteUserRegistration(uid));
    NN_RESULT_DO(m_Storage.Commit());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::srepo::NotifyUserRegistered(uid);
#endif

    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CancelUserRegistration(const Uid& uid))
{
    auto lock = m_Storage.AcquireWriterLock();
    m_ProfileStorage.Delete(uid);
    NN_RESULT_DO(m_UserRegistry.CancelUserRegistration(uid));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    DeleteUser(const Uid& uid))
{
    // 削除に必要な条件の検査
    //  - Open していないこと
    //  - NetworkServiceAccount が存在しないこと
    NN_RESULT_THROW_UNLESS(!m_UserStateManager.IsUserOpened(uid), ResultUserAlreadyOpened());
    bool registered;
    NN_RESULT_DO(m_BaasOp.IsRegistered(&registered, uid));
    NN_RESULT_THROW_UNLESS(!registered, ResultNetworkServiceAccountUnregistrationRequired());

    // LastOpened 削除
    Uid lastOpened;
    if (m_UserStateManager.TryGetLastOpenedUser(&lastOpened) && lastOpened == uid)
    {
        m_UserStateManager.InvalidateLastOpenedUser();
    }

    // 削除
    auto lock = m_Storage.AcquireWriterLock();
    m_ProfileStorage.Delete(uid);
    NN_RESULT_DO(m_UserRegistry.DeleteUser(uid));
    NN_RESULT_DO(m_Storage.Commit());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::srepo::NotifyUserDeleted(uid);
#endif

    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    SetUserPosition(const Uid& uid, std::int32_t position))
{
    auto lock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_UserRegistry.SetUserPosition(uid, position));
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetUserRegistrationNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier))
{
    os::SystemEventType* pEvent;
    NN_RESULT_DO(m_UserRegistry.GetUserRegistrationEvent(&pEvent));
    auto p = Factory::template CreateSharedEmplaced<detail::INotifier, detail::NotifierImpl<typename std::remove_reference<decltype(m_UserRegistry)>::type>>(
        &m_Allocator, &m_UserRegistry, pEvent);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutNotifier = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetUserStateChangeNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier))
{
    os::SystemEventType* pEvent;
    NN_RESULT_DO(m_UserStateManager.GetUserStateChangeEvent(&pEvent));
    auto p = Factory::template CreateSharedEmplaced<detail::INotifier, detail::NotifierImpl<typename std::remove_reference<decltype(m_UserStateManager)>::type>>(
        &m_Allocator, &m_UserStateManager, pEvent);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutNotifier = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetProfileUpdateNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier))
{
    os::SystemEventType* pEvent;
    NN_RESULT_DO(m_ProfileStorage.GetProfileUpdateEvent(&pEvent));
    auto p = Factory::template CreateSharedEmplaced<detail::INotifier, detail::NotifierImpl<typename std::remove_reference<decltype(m_ProfileStorage)>::type>>(
        &m_Allocator, &m_ProfileStorage, pEvent);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutNotifier = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetBaasUserAvailabilityChangeNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier))
{
    os::SystemEventType* pEvent;
    NN_RESULT_DO(m_BaasOp.GetBaasUserAvailabilityChangeNotifier(&pEvent));
    auto p = Factory::template CreateSharedEmplaced<detail::INotifier, baas::BaasUserInfoHolderNotifier>(&m_Allocator, m_BaasOp, pEvent);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutNotifier = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetUserLastOpenedApplication(sf::Out<ApplicationId> pOutApplicationId, sf::Out<uint32_t> pOutApplicationVersion, const Uid& user) const)
{
    NN_RESULT_THROW_UNLESS(user, ResultInvalidUserId());
    auto b = m_Usage.userOpenHistory.Find(pOutApplicationId.GetPointer(), pOutApplicationVersion.GetPointer(), user);
    NN_RESULT_THROW_UNLESS(b, ResultInvalidApplication());
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    DebugInvalidateUserResourceCache(const Uid& uid))
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    // UID に関連するリソースキャッシュをすべて破棄する
    m_BaasOp.InvalidateTokenCache(uid);
    m_BaasOp.InvalidateUserResourceCache(uid);
    m_BaasOp.InvalidateUserServiceEntryRequirementCache(uid);
    m_NasOp.InvalidateUserResourceCacheTimestamp(uid);
    m_NasOp.InvalidateNetworkServiceLicenseCache(uid);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    DebugSetUserStateOpen(const Uid& uid))
{
    ApplicationId apps[detail::ApplicationCountMax];
    int ct = m_UserStateManager.ListApplications(apps, std::extent<decltype(apps)>::value);
    NN_RESULT_THROW_UNLESS(ct > 0, ResultInvalidProtocolAccess());

    user::UserRef ref;
    NN_RESULT_DO(m_UserRegistry.GetUserRef(&ref, uid));
    return m_UserStateManager.SetUserStateOpened(apps[0], std::move(ref));
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    DebugSetUserStateClose(const Uid& uid))
{
    ApplicationId apps[detail::ApplicationCountMax];
    int ct = m_UserStateManager.ListApplications(apps, std::extent<decltype(apps)>::value);
    NN_RESULT_THROW_UNLESS(ct > 0, ResultInvalidProtocolAccess());
    return m_UserStateManager.SetUserStateClosed(apps[0], uid);
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetProfile(sf::Out<sf::SharedPointer<profile::IProfile>> pOutProfile, const Uid& uid))
{
    user::UserRef user;
    NN_RESULT_DO(m_UserRegistry.GetUserRef(&user, uid));
    auto p = Factory::template CreateSharedEmplaced<profile::IProfile, profile::ProfileEditorImpl>(&m_Allocator, std::move(user), m_ProfileStorage);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    *pOutProfile = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetProfileDigest(sf::Out<ProfileDigest> pOut, const Uid& uid))
{
    return m_ProfileStorage.GetDigest(pOut.GetPointer(), uid);
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetProfileEditor(sf::Out<sf::SharedPointer<profile::IProfileEditor>> pOutProfile, const Uid& uid))
{
    user::UserRef user;
    auto r = m_UserRegistry.GetUserRef(&user, uid);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultUserNotExist::Includes(r), r);
    sf::EmplacedRef<profile::IProfileEditor, profile::ProfileEditorImpl> p;
    if (r.IsSuccess())
    {
        p = Factory::template CreateSharedEmplaced<profile::IProfileEditor, profile::ProfileEditorImpl>(&m_Allocator, std::move(user), m_ProfileStorage, m_BaasOp);
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    }
    else
    {
        p = Factory::template CreateSharedEmplaced<profile::IProfileEditor, profile::ProfileEditorImpl>(&m_Allocator, uid, m_ProfileStorage);
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    }
    *pOutProfile = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    AuthenticateServiceAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut))
{
    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, ndas::DeviceReAuthenticationContext>(
        &m_Allocator, m_NdasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CreateFloatingRegistrationRequest(sf::Out<sf::SharedPointer<baas::IFloatingRegistrationRequest>> pOutRequest, sf::NativeHandle&& transferMemoryHandle, uint32_t size))
{
    auto p = Factory::template CreateSharedEmplaced<baas::IFloatingRegistrationRequest, baas::FloatingRegistrationRequestWithNintendoAccountImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_UserRegistry, m_ProfileStorage, m_NasSessionPool, m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(m_NdasOp, m_BaasOp, m_NasOp, std::move(transferMemoryHandle), static_cast<size_t>(size)));
    *pOutRequest = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ProxyProcedureForFloatingRegistrationWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForExternalNsa>> pOut, const detail::Uuid& sessionId))
{
    auto p = Factory::template CreateSharedEmplaced<nas::IOAuthProcedureForExternalNsa, nas::NintendoAccountFloatingRegistrationProcedureImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_Executor, m_NasSessionPool);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(sessionId));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}
NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ProxyProcedureForGuestLoginWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForExternalNsa>> pOut, const detail::Uuid& sessionId))
{
    auto p = Factory::template CreateSharedEmplaced<nas::IOAuthProcedureForExternalNsa, nas::NintendoAccountGuestLoginProcedureImpl<Allocator>>(
        &m_Allocator, m_Allocator, m_Executor, m_NasSessionPool);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(sessionId));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    TrySelectUserWithoutInteraction(sf::Out<Uid> pOut, bool isNetworkServiceAccountRequired))
{
    bool setInvalid = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (setInvalid)
        {
            *pOut = InvalidUid;
        }
    };

    // ユーザーアカウントが 1 つでない場合 Invalid を返して成功
    NN_RESULT_THROW_UNLESS(
        m_UserRegistry.GetUserCount() == 1,
        ResultSuccess());

    // スキップ設定がオフの場合 Invalid を返して成功
    settings::system::AccountSettings cfg;
    settings::system::GetAccountSettings(&cfg);
    NN_RESULT_THROW_UNLESS(
        cfg.userSelectorSettings.flags.Test<settings::system::UserSelectorFlag::SkipsIfSingleUser>(),
        ResultSuccess());

    Uid users[UserCountMax];
    NN_RESULT_DO(m_UserRegistry.ListAllUsers(users, UserCountMax));
    NN_SDK_ASSERT(users[0]);

    if (isNetworkServiceAccountRequired)
    {
        auto r = m_BaasOp.CheckAvailability(users[0]);
        NN_RESULT_THROW_UNLESS(
            r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r),
            r);
        // Unavailable の場合 Invalid を返して成功
        NN_RESULT_THROW_UNLESS(
            r.IsSuccess(),
            ResultSuccess());
    }

    *pOut = users[0];
    setInvalid = false;
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    IsUserRegistrationRequestPermitted(sf::Out<bool> pOut, const Bit64& pid) const)
{
    NN_UNUSED(pid);
    // アプリケーションが動作している場合は許可しない
    *pOut = (m_Usage.applicationCount.Get() == 0);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CheckNetworkServiceAvailabilityAsyncImpl(sf::SharedPointer<detail::IAsyncContext>* ppOutContext, const detail::ApplicationInfo& appInfo))
{
    NN_SDK_ASSERT(appInfo);

    Uid users[UserCountMax];
    NN_RESULT_DO(m_UserRegistry.ListAllUsers(users, std::extent<decltype(users)>::value));

    auto p = Factory::template CreateSharedEmplaced<detail::IAsyncContext, detail::NetworkServiceAvailabilityCheckContext>(
        &m_Allocator, appInfo, users, m_BaasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&m_Executor));
    *ppOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CheckNetworkServiceAvailabilityAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const SystemProgramIdentification& identification, const Bit64& pid))
{
    NN_UNUSED(pid);
    detail::ApplicationInfo appInfo = detail::ApplicationInfo::Get(identification.id.value, identification.version, detail::ApplicationMediaType::System);
    NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());

    sf::SharedPointer<detail::IAsyncContext> p;
    NN_RESULT_DO(CheckNetworkServiceAvailabilityAsyncImpl(&p, appInfo));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    StoreSaveDataThumbnail(const Uid& uid, const ApplicationId& appId, const sf::InBuffer& imageBuffer))
{
    NN_RESULT_THROW_UNLESS(appId != ApplicationId::GetInvalidId(), ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());
    NN_RESULT_THROW_UNLESS(imageBuffer.GetPointerUnsafe() != nullptr, ResultNullptr());
    NN_RESULT_THROW_UNLESS(imageBuffer.GetSize() == detail::SaveDataThumbnailSize, ResultInsufficientBuffer());

    // ヘッダと画像の書き込み
    const detail::SaveDataThumbnailHeader Header = NN_ACCOUNT_DETAIL_SAVE_DATA_THUMBNAIL_HEADER_INITIALIZER_1;
    NN_RESULT_DO(fs::WriteSaveDataThumbnailFile(appId.value, uid, &Header, sizeof(Header), imageBuffer.GetPointerUnsafe(), imageBuffer.GetSize()));

#if !defined(NN_SDK_BUILD_RELEASE)
    // ヘッダの検査
    detail::SaveDataThumbnailHeader h;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadSaveDataThumbnailFileHeader(appId.value, uid, &h, sizeof(h)));
    NN_ABORT_UNLESS(h.magic == detail::SaveDataThumbnailHeaderMagicNumber);
    NN_ABORT_UNLESS(h.version == detail::SaveDataThumbnailHeaderVersion_1);
#endif

    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ClearSaveDataThumbnail(const Uid& uid, const ApplicationId& appId))
{
    NN_RESULT_THROW_UNLESS(appId != ApplicationId::GetInvalidId(), ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    // ヘッダの書き込み
    const detail::SaveDataThumbnailHeader Header = NN_ACCOUNT_DETAIL_SAVE_DATA_THUMBNAIL_HEADER_INITIALIZER_0;
    NN_RESULT_DO(fs::WriteSaveDataThumbnailFileHeader(appId.value, uid, &Header, sizeof(Header)));

#if !defined(NN_SDK_BUILD_RELEASE)
    // ヘッダの検査
    detail::SaveDataThumbnailHeader h;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadSaveDataThumbnailFileHeader(appId.value, uid, &h, sizeof(h)));
    NN_ABORT_UNLESS(h.magic == detail::SaveDataThumbnailHeaderMagicNumber);
    NN_ABORT_UNLESS(h.version == detail::SaveDataThumbnailHeaderVersion_0);
#endif

    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    LoadSaveDataThumbnail(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOut, const Uid& uid, const ApplicationId& appId))
{
    NN_RESULT_THROW_UNLESS(appId != ApplicationId::GetInvalidId(), ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());
    NN_RESULT_THROW_UNLESS(pOut.GetPointerUnsafe() != nullptr, ResultNullptr());

    // ヘッダの検査
    detail::SaveDataThumbnailHeader header;
    NN_RESULT_DO(fs::ReadSaveDataThumbnailFileHeader(appId.value, uid, &header, sizeof(header)));
    NN_RESULT_THROW_UNLESS(header.magic == detail::SaveDataThumbnailHeaderMagicNumber, ResultSaveDataThumbnailEmpty());

    // 読み出すサイズの決定
    size_t sizeToRead;
    switch (header.version)
    {
    case detail::SaveDataThumbnailHeaderVersion_0:
        // 明示的にクリアされている
        NN_RESULT_THROW(ResultSaveDataThumbnailEmpty());

    case detail::SaveDataThumbnailHeaderVersion_1:
        sizeToRead = detail::SaveDataThumbnailSize;
        break;

    default:
        NN_RESULT_THROW(ResultSaveDataThumbnailEmpty());
    }
    NN_RESULT_THROW_UNLESS(pOut.GetSize() >= sizeToRead, ResultInsufficientBuffer());

    // 読み出し
    NN_RESULT_DO(fs::ReadSaveDataThumbnailFile(appId.value, uid, &header, sizeof(header), pOut.GetPointerUnsafe(), sizeToRead));
    *pOutActualSize = static_cast<uint32_t>(sizeToRead); // サイズ固定なのでチェックしない
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetSaveDataThumbnailExistence(sf::Out<bool> pOut, const Uid& uid, const ApplicationId& appId))
{
    NN_RESULT_THROW_UNLESS(appId != ApplicationId::GetInvalidId(), ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());
    NN_RESULT_THROW_UNLESS(pOut.GetPointer() != nullptr, ResultNullptr());

    // ヘッダの検査
    detail::SaveDataThumbnailHeader header;
    auto r = fs::ReadSaveDataThumbnailFileHeader(appId.value, uid, &header, sizeof(header));
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultTargetNotFound::Includes(r), r);
    if (!(r.IsSuccess() && header.magic == detail::SaveDataThumbnailHeaderMagicNumber))
    {
        *pOut = false;
        NN_RESULT_SUCCESS;
    }
    switch (header.version)
    {
    case detail::SaveDataThumbnailHeaderVersion_1:
        *pOut = true;
        NN_RESULT_SUCCESS;

    case detail::SaveDataThumbnailHeaderVersion_0:
        // 明示的にクリアされている
    default:
        *pOut = false;
        NN_RESULT_SUCCESS;
    }
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ActivateOpenContextRetention(sf::Out<sf::SharedPointer<detail::ISessionObject>> pOut, const ApplicationId& appId))
{
    auto p = Factory::template CreateSharedEmplaced<detail::ISessionObject, detail::OpenContextRetentionActivator<Allocator>>(
        &m_Allocator, m_UserStateManager, m_OpenContextRetainer);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(appId));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    DebugActivateOpenContextRetention(sf::Out<sf::SharedPointer<detail::ISessionObject>> pOut))
{
    auto p = Factory::template CreateSharedEmplaced<detail::ISessionObject, detail::OpenContextRetentionActivatorForDebug<Allocator>>(
        &m_Allocator, m_UserStateManager, m_OpenContextRetainer);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize());
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ListQualifiedUsers(const sf::OutArray<Uid>& outUsers, const ApplicationId& appId))
{
    NN_SDK_ASSERT(outUsers.GetLength() <= static_cast<uint64_t>(std::numeric_limits<int>::max()));

    Uid users[UserCountMax];
    NN_RESULT_DO(m_UserRegistry.ListAllUsers(users, std::extent<decltype(users)>::value));

    auto userCount = static_cast<int>(std::count_if(users, users + std::extent<decltype(users)>::value, [](const auto& u) -> bool { return u; }));
    auto qualifiedCount = m_UserStateManager.SelectQualifiedUsers(
        outUsers.GetData(), static_cast<int>(outUsers.GetLength()),
        appId, users, userCount);
    for (auto i = qualifiedCount; i < static_cast<int>(outUsers.GetLength()); ++ i)
    {
        outUsers.GetData()[i] = InvalidUid;
    }
    NN_RESULT_SUCCESS;
}
#undef NN_ACCOUNT_DEFINE_SERVICE_METHOD

/* AccountServiceImplForApplication -------------------------------------------------------- */


template <typename Allocator>
AccountServiceImplForApplication<Allocator>::AccountServiceImplForApplication(
    Allocator& allocator,
    detail::Usage& usage,
    const detail::AbstractLocalStorage& storage,
    user::UserRegistry& userRegistry,
    user::UserStateManager& userStateManager,
    detail::OpenContextRetainer<Allocator>& openContextRetainer,
    profile::ProfileStorage& profileStorage,
    nas::NasSessionPool<Allocator>& nasSessionPool,
    ndas::NdasOperator& ndasOp,
    baas::BaasOperator& baasOp,
    nas::NasOperator& nasOp,
    Executor& executor) NN_NOEXCEPT
    : AccountServiceImpl<Allocator>(allocator, usage, storage, userRegistry, userStateManager, openContextRetainer, profileStorage, nasSessionPool, ndasOp, baasOp, nasOp, executor)
    , m_UserStateManagerActivator(userStateManager)
    , m_Counter(usage.applicationCount) // Application として初期化された数を数える
    , m_AppInfo(detail::InvalidApplicationInfo)
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(ovln::InitializeSenderLibraryForOverlay());
#endif
}

template <typename Allocator>
AccountServiceImplForApplication<Allocator>::~AccountServiceImplForApplication() NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    ovln::FinalizeSenderLibraryForOverlay();
#endif
}

#define NN_ACCOUNT_DEFINE_SERVICE_METHOD(rtype, methodInfo) \
template <typename Allocator> \
inline rtype AccountServiceImplForApplication<Allocator>::methodInfo NN_NOEXCEPT

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    InitializeApplicationInfoV0(const Bit64& pid))
{
    NN_RESULT_THROW_UNLESS(!m_AppInfo, ResultInvalidProtocolAccess());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    os::ProcessId processId = {pid};
    arp::ApplicationLaunchProperty launchProperty;
    auto r = arp::GetApplicationLaunchProperty(&launchProperty, processId);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || arp::ResultNotRegistered::Includes(r), r);
    if (!r.IsSuccess())
    {
        // 見つからなかった場合呼び出しもとに返してハンドリングする。
        m_AppInfo = detail::InvalidApplicationInfo;
        NN_RESULT_THROW(ResultInvalidApplication());
    }

    detail::ApplicationInfo appInfo = detail::InvalidApplicationInfo;
    appInfo.launchProperty = launchProperty;
    switch (launchProperty.storageId)
    {
    case ncm::StorageId::Card:
        appInfo.mediaType = detail::ApplicationMediaType::GameCard;
        break;
    case ncm::StorageId::BuildInUser:
    case ncm::StorageId::Host:
    case ncm::StorageId::SdCard:
        appInfo.mediaType = detail::ApplicationMediaType::Digital;
        break;
    default:
        NN_RESULT_THROW(ResultInvalidApplication());
    }
    NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
    m_AppInfo = appInfo;

#else
    NN_UNUSED(pid);
    m_AppInfo = detail::ApplicationInfo::Get(0x0100f80000492000ull, 0, detail::ApplicationMediaType::Digital);
#endif

    NN_RESULT_DO(m_UserStateManagerActivator.Initialize({m_AppInfo.launchProperty.id.value}));
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    InitializeApplicationInfo(const Bit64& pid))
{
    NN_RESULT_DO(InitializeApplicationInfoV0(pid));
    switch (m_AppInfo.mediaType)
    {
    case detail::ApplicationMediaType::Digital:
        {
            Uid users[UserCountMax];
            int count;
            auto r = detail::ListUsersWithRightsRestriction(
                &count, users, std::extent<decltype(users)>::value,
                m_AppInfo.launchProperty,
                this->m_UserRegistry, this->m_BaasOp);
            if (!r.IsSuccess())
            {
                NN_RESULT_THROW_UNLESS(ResultUserQualificationNotLimited::Includes(r), r);
                NN_DETAIL_ACCOUNT_TRACE("User qualification is not limited for 0x%016llx on %d\n", m_AppInfo.launchProperty.id.value, m_AppInfo.mediaType);
                break;
            }
            this->m_UserStateManager.EnableQualificationLimitation({m_AppInfo.launchProperty.id.value}, users, count);
        }
        break;
    default:
        NN_DETAIL_ACCOUNT_TRACE("All users are qualified on this media %d for 0x%016llx\n", m_AppInfo.mediaType, m_AppInfo.launchProperty.id.value);
        break;
    }
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    GetBaasAccountManagerForApplication(sf::Out<sf::SharedPointer<baas::IManagerForApplication>> pOutManager, const Uid& uid))
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());

    user::UserRef user;
    NN_RESULT_DO(ServiceBase::m_UserRegistry.GetUserRef(&user, uid));
    auto p = ServiceBase::Factory::template CreateSharedEmplaced<baas::IManagerForApplication, detail::OpenContextImpl<Allocator>>(
        &(this->m_Allocator),
        typename detail::OpenContextImpl<Allocator>::ParentPtr(this, true),
        this->m_Allocator, this->m_UserStateManager, this->m_NasSessionPool, this->m_BaasOp, this->m_NasOp, std::move(user), m_AppInfo, this->m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());

    NN_RESULT_DO(p.GetImpl().Initialize());
    this->m_Usage.userOpenHistory.Add(uid, {m_AppInfo.launchProperty.id.value}, m_AppInfo.launchProperty.version);

    *pOutManager = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    AuthenticateApplicationAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut))
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());

    auto p = ServiceBase::Factory::template CreateSharedEmplaced<detail::IAsyncContext, ndas::ApplicationReAuthenticationContext>(
        &this->m_Allocator, m_AppInfo, this->m_NdasOp);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(&this->m_Executor));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CheckNetworkServiceAvailabilityAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext))
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());

    sf::SharedPointer<detail::IAsyncContext> p;
    NN_RESULT_DO(this->CheckNetworkServiceAvailabilityAsyncImpl(&p, m_AppInfo));
    *pOutContext = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    CreateGuestLoginRequest(sf::Out<sf::SharedPointer<baas::IGuestLoginRequest>> pOutRequest, sf::NativeHandle&& transferMemoryHandle, uint32_t size))
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());

    auto p = ServiceBase::Factory::template CreateSharedEmplaced<baas::IGuestLoginRequest, baas::GuestLoginRequestWithNintendoAccountImpl<Allocator>>(
        &this->m_Allocator, this->m_Allocator, this->m_NasSessionPool, this->m_Executor);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
    NN_RESULT_DO(p.GetImpl().Initialize(m_AppInfo, this->m_NdasOp, this->m_BaasOp, this->m_NasOp, std::move(transferMemoryHandle), static_cast<size_t>(size)));
    *pOutRequest = std::move(p);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    IsUserRegistrationRequestPermitted(sf::Out<bool> pOut, const Bit64& pid) const)
{
    NN_UNUSED(pid);
    // 複数のアプリケーションが動作している場合は許可しない
    *pOut = (this->m_Usage.applicationCount.Get() == 1);
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    StoreOpenContext(sf::SharedPointer<detail::OpenContextImpl<Allocator>>&& ptr))
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(
        this->m_OpenContextRetainer.TryPush(std::move(ptr)),
        ResultInvalidProtocolAccess());
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    LoadOpenContext(sf::Out<sf::SharedPointer<baas::IManagerForApplication>> outPtr, const Uid& uid))
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());
    auto ptr = this->m_OpenContextRetainer.Pull({m_AppInfo.launchProperty.id.value}, uid);
    NN_RESULT_THROW_UNLESS(ptr, ResultInvalidProtocolAccess());

    *outPtr = ServiceBase::Factory::template CreateShared<baas::IManagerForApplication>(
        &this->m_Allocator, std::move(ptr));
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ListOpenContextStoredUsers(sf::OutArray<Uid> outUids) const)
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());
    int count = this->m_OpenContextRetainer.ListUsers(outUids.GetData(), static_cast<int>(outUids.GetLength()), {m_AppInfo.launchProperty.id.value});
    for (auto i = count; i < static_cast<int>(outUids.GetLength()); ++ i)
    {
        outUids.GetData()[i] = InvalidUid;
    }
    NN_RESULT_SUCCESS;
}

NN_ACCOUNT_DEFINE_SERVICE_METHOD(
    Result,
    ListQualifiedUsers(const sf::OutArray<Uid>& outUsers))
{
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultInvalidProtocolAccess());
    return ServiceBase::ListQualifiedUsers(outUsers, {m_AppInfo.launchProperty.id.value});
}

#undef NN_ACCOUNT_DEFINE_SERVICE_METHOD

}} // ~namespace nn::account
