﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_Interface.sfdl.h>
#include <nn/account/detail/account_INotifier.sfdl.h>
#include <nn/account/detail/account_ISessionObject.sfdl.h>
#include <nn/account/nas/account_Interface.sfdl.h>
#include <nn/account/profile/account_Interface.sfdl.h>
#include <nn/account/account_IAccountService.sfdl.h>

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/detail/account_ReferenceContext.h>
#include <nn/account/detail/account_Usage.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/ndas/account_NdasOperator.h>
#include <nn/account/profile/account_ProfileStorage.h>
#include <nn/account/user/account_UserRegistry.h>
#include <nn/account/user/account_UserStateManager.h>
#include <nn/account/account_Types.h>

#include <nn/nn_Result.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace account {
class Executor;
}} // ~namespace nn::account

namespace nn { namespace account {
namespace detail {
class UserStateManagerActivator
{
    user::UserStateManager& m_UserStateManager;
    bool m_Initialized {false};
    ApplicationId m_ApplicationId;

public:
    explicit UserStateManagerActivator(user::UserStateManager& userStateManager) NN_NOEXCEPT;
    ~UserStateManagerActivator() NN_NOEXCEPT;
    Result Initialize(ApplicationId appId) NN_NOEXCEPT;
};
} // ~nn::account::detail

template <typename Allocator>
class AccountServiceImpl
    : public sf::ISharedObject
{
protected:
    typedef sf::ObjectFactory<typename Allocator::Policy> Factory;

    Allocator& m_Allocator;
    detail::Usage& m_Usage;

    const detail::AbstractLocalStorage& m_Storage;
    user::UserRegistry& m_UserRegistry;
    user::UserStateManager& m_UserStateManager;
    detail::OpenContextRetainer<Allocator>& m_OpenContextRetainer;
    profile::ProfileStorage& m_ProfileStorage;
    nas::NasSessionPool<Allocator>& m_NasSessionPool;
    ndas::NdasOperator& m_NdasOp;
    baas::BaasOperator& m_BaasOp;
    nas::NasOperator& m_NasOp;
    Executor& m_Executor;

    Result CheckNetworkServiceAvailabilityAsyncImpl(sf::SharedPointer<detail::IAsyncContext>* ppOutContext, const detail::ApplicationInfo& appInfo) NN_NOEXCEPT;

public:
    AccountServiceImpl(
        Allocator& allocator,
        detail::Usage& usage,
        const detail::AbstractLocalStorage& storage,
        user::UserRegistry& userRegistry,
        user::UserStateManager& userStateManager,
        detail::OpenContextRetainer<Allocator>& openContextHolder,
        profile::ProfileStorage& profileStorage,
        nas::NasSessionPool<Allocator>& nasSessionPool,
        ndas::NdasOperator& ndasOp,
        baas::BaasOperator& baasOp,
        nas::NasOperator& nasOp,
        Executor& executor) NN_NOEXCEPT
        : m_Allocator(allocator)
        , m_Usage(usage)
        , m_Storage(storage)
        , m_UserRegistry(userRegistry)
        , m_UserStateManager(userStateManager)
        , m_OpenContextRetainer(openContextHolder)
        , m_ProfileStorage(profileStorage)
        , m_NasSessionPool(nasSessionPool)
        , m_NdasOp(ndasOp)
        , m_BaasOp(baasOp)
        , m_NasOp(nasOp)
        , m_Executor(executor)
    {
    }

    Result GetBaasAccountManagerForApplication(sf::Out<sf::SharedPointer<baas::IManagerForApplication>> pOutManager, const Uid& uid, const Bit64& pid) NN_NOEXCEPT;
    Result GetBaasAccountManagerForSystemService(sf::Out<sf::SharedPointer<baas::IManagerForSystemService>> pOutManager, const Uid& uid) NN_NOEXCEPT;
    Result GetBaasAccountAdministrator(sf::Out<sf::SharedPointer<baas::IAdministrator>> pOutAdministrator, const Uid& uid) NN_NOEXCEPT;

    Result GetUserCount(sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT;
    Result GetUserExistence(sf::Out<bool> pOutExistence, const Uid& uid) NN_NOEXCEPT;
    Result ListAllUsers(const sf::OutArray<Uid>& outUsers) NN_NOEXCEPT;
    Result ListOpenUsers(const sf::OutArray<Uid>& outUsers) NN_NOEXCEPT;
    Result GetLastOpenedUser(sf::Out<Uid> pOutUid) NN_NOEXCEPT;

    Result BeginUserRegistration(sf::Out<Uid> pOutUid) NN_NOEXCEPT;
    Result CompleteUserRegistration(const Uid& uid) NN_NOEXCEPT;
    Result CompleteUserRegistrationForcibly(const Uid& uid) NN_NOEXCEPT;
    Result CancelUserRegistration(const Uid& uid) NN_NOEXCEPT;
    Result DeleteUser(const Uid& uid) NN_NOEXCEPT;
    Result SetUserPosition(const Uid& uid, std::int32_t position) NN_NOEXCEPT;

    Result GetUserRegistrationNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier) NN_NOEXCEPT;
    Result GetUserStateChangeNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier) NN_NOEXCEPT;
    Result GetProfileUpdateNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier) NN_NOEXCEPT;
    Result GetBaasUserAvailabilityChangeNotifier(sf::Out<sf::SharedPointer<detail::INotifier>> pOutNotifier) NN_NOEXCEPT;

    Result GetUserLastOpenedApplication(sf::Out<ApplicationId> pOutApplicationId, sf::Out<uint32_t> pOutApplicationVersion, const Uid& user) const NN_NOEXCEPT;

    Result DebugInvalidateUserResourceCache(const Uid& uid) NN_NOEXCEPT;
    Result DebugSetUserStateOpen(const Uid& uid) NN_NOEXCEPT;
    Result DebugSetUserStateClose(const Uid& uid) NN_NOEXCEPT;

    Result GetProfile(sf::Out<sf::SharedPointer<profile::IProfile>> pOutProfile, const Uid& uid) NN_NOEXCEPT;
    Result GetProfileDigest(sf::Out<ProfileDigest> pOut, const Uid& uid) NN_NOEXCEPT;
    Result GetProfileEditor(sf::Out<sf::SharedPointer<profile::IProfileEditor>> pOutProfile, const Uid& uid) NN_NOEXCEPT;

    Result AuthenticateServiceAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT;

    Result CreateFloatingRegistrationRequest(sf::Out<sf::SharedPointer<baas::IFloatingRegistrationRequest>> pOutRequest, sf::NativeHandle&& transferMemoryHandle, uint32_t size) NN_NOEXCEPT;
    Result ProxyProcedureForFloatingRegistrationWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForExternalNsa>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT;
    Result ProxyProcedureForGuestLoginWithNintendoAccount(sf::Out<sf::SharedPointer<nas::IOAuthProcedureForExternalNsa>> pOut, const detail::Uuid& sessionId) NN_NOEXCEPT;

    Result TrySelectUserWithoutInteraction(sf::Out<Uid> pOut, bool isNetworkServiceAccountRequired) NN_NOEXCEPT;
    Result IsUserRegistrationRequestPermitted(sf::Out<bool> pOut, const Bit64& pid) const NN_NOEXCEPT;

    Result CheckNetworkServiceAvailabilityAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext, const SystemProgramIdentification& identification, const Bit64& pid) NN_NOEXCEPT;

    Result LoadSaveDataThumbnail(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOut, const Uid& uid, const ApplicationId& appId) NN_NOEXCEPT;
    Result StoreSaveDataThumbnail(const Uid& uid, const ApplicationId& appId, const sf::InBuffer& imageBuffer) NN_NOEXCEPT;
    Result ClearSaveDataThumbnail(const Uid& uid, const ApplicationId& appId) NN_NOEXCEPT;
    Result GetSaveDataThumbnailExistence(sf::Out<bool> pOut, const Uid& uid, const ApplicationId& appId) NN_NOEXCEPT;

    Result ActivateOpenContextRetention(sf::Out<sf::SharedPointer<detail::ISessionObject>> pOut, const ApplicationId& appId) NN_NOEXCEPT;
    Result DebugActivateOpenContextRetention(sf::Out<sf::SharedPointer<detail::ISessionObject>> pOut) NN_NOEXCEPT;

    Result ListQualifiedUsers(const sf::OutArray<Uid>& outUsers, const ApplicationId& appId) NN_NOEXCEPT;
};

template <typename Allocator>
class AccountServiceImplForApplication
    : public AccountServiceImpl<Allocator>
{
private:
    typedef AccountServiceImpl<Allocator> ServiceBase;

    detail::UserStateManagerActivator m_UserStateManagerActivator;
    detail::SharedCounter m_Counter;
    detail::ApplicationInfo m_AppInfo;

public:
    AccountServiceImplForApplication(
        Allocator& allocator,
        detail::Usage& usage,
        const detail::AbstractLocalStorage& storage,
        user::UserRegistry& userRegistry,
        user::UserStateManager& userStateManager,
        detail::OpenContextRetainer<Allocator>& openContextHolder,
        profile::ProfileStorage& profileStorage,
        nas::NasSessionPool<Allocator>& nasSessionPool,
        ndas::NdasOperator& ndasOp,
        baas::BaasOperator& baasOp,
        nas::NasOperator& nasOp,
        Executor& executor) NN_NOEXCEPT;
    ~AccountServiceImplForApplication() NN_NOEXCEPT;

    Result InitializeApplicationInfoV0(const Bit64& pid) NN_NOEXCEPT;
    Result InitializeApplicationInfo(const Bit64& pid) NN_NOEXCEPT;

    Result GetBaasAccountManagerForApplication(sf::Out<sf::SharedPointer<baas::IManagerForApplication>> pOutManager, const Uid& uid) NN_NOEXCEPT;
    Result AuthenticateApplicationAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOut) NN_NOEXCEPT;
    Result CreateGuestLoginRequest(sf::Out<sf::SharedPointer<baas::IGuestLoginRequest>> pOutRequest, sf::NativeHandle&& transferMemoryHandle, uint32_t size) NN_NOEXCEPT;

    Result IsUserRegistrationRequestPermitted(sf::Out<bool> pOut, const Bit64& pid) const NN_NOEXCEPT;

    Result CheckNetworkServiceAvailabilityAsync(sf::Out<sf::SharedPointer<detail::IAsyncContext>> pOutContext) NN_NOEXCEPT;

    Result StoreSaveDataThumbnail(const Uid& uid, const sf::InBuffer& imageBuffer) NN_NOEXCEPT
    {
        ApplicationId appId = {m_AppInfo.launchProperty.id.value};
        return ServiceBase::StoreSaveDataThumbnail(uid, appId, imageBuffer);
    }
    Result ClearSaveDataThumbnail(const Uid& uid) NN_NOEXCEPT
    {
        ApplicationId appId = {m_AppInfo.launchProperty.id.value};
        return ServiceBase::ClearSaveDataThumbnail(uid, appId);
    }

    Result StoreOpenContext(sf::SharedPointer<detail::OpenContextImpl<Allocator>>&& ptr) NN_NOEXCEPT;
    Result LoadOpenContext(sf::Out<sf::SharedPointer<baas::IManagerForApplication>> outPtr, const Uid& uid) NN_NOEXCEPT;
    Result ListOpenContextStoredUsers(sf::OutArray<Uid> outUids) const NN_NOEXCEPT;

    Result ListQualifiedUsers(const sf::OutArray<Uid>& outUsers) NN_NOEXCEPT;
};

}} // ~namespace nn::account

// 実装部分のインクルード
#include <nn/account/account_AccountServiceImpl_impl.h>
