﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Nintendo.ServiceFramework.Validation
{
    using Localization;

    public class InterfaceValidator
    {
        /// <summary>
        /// 与えられた型がサービスインターフェイスとして正しいかどうかを validate する
        /// </summary>
        /// <param name="t">validate する対象の型</param>
        public void Validate(Type t)
        {
            if (t.IsGenericTypeDefinition)
            {
                throw new InvalidInterfaceException(@"SF-IF-NotSupported:GenericTypeDefinition", t, "ジェネリック型定義はサポートされていません".Ja());
            }
            if (t.IsGenericType)
            {
                throw new InvalidInterfaceException(@"SF-IF-NotSupported:GenericInterface", t, "ジェネリックインターフェイスはサポートされていません".Ja());
            }
            if (t.GetInterface(typeof(nn.sf.IServiceObject).FullName) == null)
            {
                throw new InvalidInterfaceException(@"SF-IF-InvalidBaseInterface:NotIServiceObject", t, "{0} を継承している必要があります".Ja(), typeof(nn.sf.IServiceObject));
            }
            if (!t.IsSingleInheritedInterface())
            {
                throw new InvalidInterfaceException(@"SF-IF-InvalidBaseInterface:MultipleBase", t, "インターフェースの多重継承はサポートされていません".Ja());
            }
            CheckNoSameNameMethod(t);
            foreach (var m in t.GetMethods())
            {
                ValidateMethod(m);
            }
        }

        private static void CheckNoSameNameMethod(Type t)
        {
            var methodNames = new HashSet<string>();
            var methodIds = new HashSet<int>();
            foreach (var m in t.GetAllInterfaceMethods())
            {
                var methodName = m.Name;
                if (methodNames.Contains(methodName))
                {
                    throw new InvalidInterfaceException(@"SF-IF-MethodNameDuplicated", t, "メソッド名 {0} を持つメソッドが複数存在します".Ja(), methodName);
                }
                methodNames.Add(methodName);
                var id = MethodIdAttribute.GetId(m);
                if (id.HasValue)
                {
                    if (methodIds.Contains(id.Value))
                    {
                        throw new InvalidInterfaceException(@"SF-IF-MethodIdDuplicated", t, "メソッド ID = {0} を持つメソッドが複数存在します".Ja(), id.Value.ToString());
                    }
                    methodIds.Add(id.Value);
                }
            }
        }

        private static bool IsSupportedReturnType(Type t)
        {
            return false
                || SfBuiltIn.CanAccept(t)
                || SfNnResult.CanAccept(t)
                || SfStruct.CanAccept(t)
                || SfInterface.CanAccept(t)
                || t == typeof(void);
        }

        private void ValidateMethod(MethodInfo m)
        {
            if (!IsSupportedReturnType(m.ReturnType))
            {
                throw new InvalidMethodException(@"SF-IF-NotSupported:ReturnType", m, "{0} は返り値型としてサポートされていません".Ja(), m.ReturnType);
            }
            if (!MethodIdAttribute.GetId(m).HasValue)
            {
                throw new InvalidMethodException(@"SF-IF-NoMethodId", m, "MethodId が指定されていません".Ja());
            }
            foreach (var p in m.GetParameters())
            {
                ValidateParameter(p);
            }
        }

        private void ValidateParameter(ParameterInfo p)
        {
            if (p.IsOut)
            {
                throw new InvalidMethodParameterException(@"SF-IF-NotSupported:NativeOutParameter", p, "out パラメータは使用できません。{0} を使用してください".Ja(), typeof(nn.sf.Out<>));
            }
            var pt = p.ParameterType;
            {
                var et = pt.GetSfParameterElementType();
                var move = p.GetCustomAttribute<MoveHandleAttribute>();
                if (SfNativeHandle.CanAccept(et))
                {
                    // nop
                }
                else
                {
                    if (!(move == null))
                    {
                        throw new InvalidMethodParameterException(
                            @"SF-IF-InvalidParameterAttribute",
                            p,
                            "ネイティブハンドルパラメータ以外には、{0} を指定することはできません。".Ja(),
                            typeof(MoveHandleAttribute).FullName);
                    }
                }
                var procId = p.GetCustomAttribute<Hipc.ProcessIdAttribute>();
                if (!(procId == null))
                {
                    if (et.Equals(typeof(nn.Bit64)))
                    {
                        if (pt.GetInOutType() != InOutType.In)
                        {
                            throw new InvalidMethodParameterException(
                                @"SF-IF-InvalidParameterType:HipcProcessId",
                                p,
                                "入力パラメータ以外には、 {0} を指定することはできません。".Ja(),
                                typeof(Hipc.ProcessIdAttribute).FullName);
                        }
                    }
                    else
                    {
                        throw new InvalidMethodParameterException(
                            @"SF-IF-InvalidParameterAttribute",
                            p,
                            "nn.Bit64 以外には、 {0} を指定することはできません。".Ja(),
                            typeof(Hipc.ProcessIdAttribute).FullName);
                    }
                }
                var largeData = p.GetCustomAttribute<LargeDataAttribute>();
                if (largeData != null)
                {
                    if (!(et.GetSfEntity() is SfValueType))
                    {
                        throw new InvalidMethodParameterException(
                            @"SF-IF-InvalidParameterAttribute",
                            p,
                            "値型以外には、{0} を指定することはできません。".Ja(),
                            typeof(LargeDataAttribute).FullName);
                    }
                }
                var hipcMapTransferSecurityAttributes = p.GetCustomAttributes<Hipc.MapTransferSecurityAttribute>().ToArray();
                if (hipcMapTransferSecurityAttributes.Length >= 1)
                {
                    if (!SfNnBuffer.CanAccept(pt))
                    {
                        throw new InvalidMethodParameterException(
                            @"SF-IF-InvalidParameterAttribute:HipcMapTransferAttribute",
                            p,
                            "バッファ型以外には、HIPC マップ転送属性を指定することはできません。".Ja());
                    }
                }
                if (!(hipcMapTransferSecurityAttributes.Length <= 1))
                {
                    throw new InvalidMethodParameterException(
                        @"SF-IF-InvalidParameterAttribute:HipcMapTransferAttribute",
                        p,
                        "複数の HIPC マップ転送属性が指定されています。".Ja());
                }
                var pres = p.GetCustomAttributes<CppCode.PreBaseAttribute>().ToArray();
                if (pres.Length >= 1)
                {
                    if (!(pt.GetInOutType() == InOutType.In))
                    {
                        throw new InvalidMethodParameterException(
                            @"SF-IF-InvalidParameterAttribute:PreAttribute",
                            p,
                            "事前条件属性は入力パラメータにしか付加できません。".Ja());
                    }
                }
            }
            if (!IsSupportedParameterType(pt))
            {
                throw new InvalidMethodParameterException(@"SF-IF-NotSupported:ParameterType", p, "{0} はパラメータ型としてサポートされていません".Ja(), p.ParameterType);
            }
        }

        private static bool IsSupportedParameterType(Type pt)
        {
            switch (pt.GetInOutType())
            {
                case InOutType.In:
                {
                    return false
                        || SfNnBuffer.CanAccept(pt)
                        || SfBuiltIn.CanAccept(pt)
                        || SfStruct.CanAccept(pt)
                        || SfInterface.CanAccept(pt)
                        || SfNativeHandle.CanAccept(pt);
                }
                case InOutType.Out:
                {
                    var et = pt.GetSfParameterElementType();
                    return false
                        || SfBuiltIn.CanAccept(et)
                        || SfStruct.CanAccept(et)
                        || SfInterface.CanAccept(et)
                        || SfNativeHandle.CanAccept(et);
                }
                default:
                {
                    throw new WrongImplementationException();
                }
            }
        }
    }
}
