﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Nintendo.ServiceFramework
{
    /// <summary>
    /// SFDL エンティティを表す型
    /// </summary>
    /// <remarks>
    /// SFDL エンティティとして有効な System.Type とインスタンスが一対一対応する。(等価である)
    /// SfEntity 同士の同一性比較は、参照比較の結果と値比較の結果は一致する。
    /// System.Type と SfEntity の間には暗黙的相互変換を用意されている。
    /// 変換に成功するとき、変換元が同一である場合、変換結果も同一である。
    /// 変換元が同一でない場合、変換結果も同一でない。
    /// また、変換結果に逆変換をした場合には、元の参照を得る。
    /// </remarks>
    public abstract partial class SfEntity
    {
        public Type InnerType { get; private set; }

        protected SfEntity(Type type)
        {
            this.InnerType = type;
        }

        /// <summary>
        /// このエンティティが参照する型の列挙を取得する
        /// </summary>
        /// <returns>参照する型の列挙(重複可)</returns>
        internal abstract IEnumerable<SfEntity> GetReferredEntities();

        /// <summary>
        /// このエンティティが強く参照する型の列挙を取得する
        /// </summary>
        /// <returns>強く参照する型の列挙(重複可)</returns>
        /// <remarks>
        /// 「強く参照する」とは、(宣言ではなく)必ず定義が先にされている必要のあるものを指す。
        /// 構造体中に構造体が配置されているような場合、外の構造体は中の構造体を強く参照する。
        /// </remarks>
        internal abstract IEnumerable<SfEntity> GetStronglyReferredEntities();

        internal virtual IEnumerable<SfEntity> GetElementEntities()
        {
            yield return this;
        }

        /// <summary>
        /// System.Type から SfEntity への暗黙変換
        /// </summary>
        /// <param name="type">System.Type 値</param>
        /// <returns>SfEntity 値</returns>
        public static implicit operator SfEntity(Type type)
        {
            return SfEntityGetter.GetSfEntity(type);
        }

        /// <summary>
        /// SfEntity から System.Type への暗黙変換
        /// </summary>
        /// <param name="entity">SfEntity 値</param>
        /// <returns>System.Type 値</returns>
        public static implicit operator Type(SfEntity entity)
        {
            return entity.InnerType;
        }
    }
}
