﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;
using System.Runtime.Serialization;

namespace Nintendo.ServiceFramework
{
    using Localization;

    /// <summary>
    /// Nintendo.ServiceFramework で起きる例外の共通基底型
    /// </summary>
    [Serializable]
    public class ServiceFrameworkException : Exception
    {
        public ServiceFrameworkException() { }
        protected ServiceFrameworkException(string message) : base(message) { }
        protected ServiceFrameworkException(string message, Exception inner) : base(message, inner) { }
        protected ServiceFrameworkException(SerializationInfo info, StreamingContext context) : base(info, context) { }
    }

    /// <summary>
    /// 実装が間違っている(バグ)場合に投げられる例外型
    /// </summary>
    [Serializable]
    public class WrongImplementationException : ServiceFrameworkException
    {
        public WrongImplementationException() { }
        public WrongImplementationException(string message) : base(message) { }
        public WrongImplementationException(string message, Exception inner) : base(message, inner) { }
        protected WrongImplementationException(SerializationInfo info, StreamingContext context) : base(info, context) { }
    }

    /// <summary>
    /// ユーザに自然言語で見せることを前提とした例外型
    /// </summary>
    [Serializable]
    public class UserErrorException : ServiceFrameworkException
    {
        /// <summary>
        /// 文字列のエラーコード(ユーザ言語非依存)
        /// </summary>
        public string ErrorCode { get; private set; }

        public UserErrorException() { }
        public UserErrorException(string errorCode, string message, Exception inner)
            : base(message, inner)
        {
            this.ErrorCode = errorCode;
        }

        private static readonly string ErrorCodeKeyName = "ErrorCodeOf" + typeof(UserErrorException).Name;

        protected UserErrorException(SerializationInfo info, StreamingContext context)
            : base(info, context)
        {
            this.ErrorCode = info.GetString(ErrorCodeKeyName);
        }

        public override void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue(ErrorCodeKeyName, ErrorCode);
            base.GetObjectData(info, context);
        }

        public override string Message
        {
            get
            {
                return string.Format(@"[{0}] {1}", ErrorCode, base.Message);
            }
        }
    }

    /// <summary>
    /// 不正な SFDL を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public class InvalidSfdlException : UserErrorException
    {
        public InvalidSfdlException() { }
        protected InvalidSfdlException(string errorCode, string message, Exception inner) : base(errorCode, message, inner) { }
        protected InvalidSfdlException(SerializationInfo info, StreamingContext context) : base(info, context) { }
    }

    /// <summary>
    /// InvalidSfdlException に特定の型の値を付加した共通例外型
    /// </summary>
    /// <typeparam name="T">付加する値の型</typeparam>
    [Serializable]
    public class InvalidSfdlExceptionWithValue<T> : InvalidSfdlException
    {
        /// <summary>
        /// 付加する値
        /// </summary>
        public T Value { get; private set; }

        public InvalidSfdlExceptionWithValue() { }
        protected InvalidSfdlExceptionWithValue(string errorCode, T value, string message) : this(errorCode, value, message, null) { }

        protected InvalidSfdlExceptionWithValue(string errorCode, T value, string message, Exception inner)
            : base(errorCode, message, inner)
        {
            this.Value = value;
        }

        private static readonly string ValueKeyName = "ValueOf" + typeof(InvalidSfdlExceptionWithValue<>).Name;

        protected InvalidSfdlExceptionWithValue(SerializationInfo info, StreamingContext context)
            : base(info, context)
        {
            this.Value = (T)info.GetValue(ValueKeyName, typeof(T));
        }

        public override void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue(ValueKeyName, Value);
            base.GetObjectData(info, context);
        }
    }

    [Serializable]
    public class UnknownEntityException : InvalidSfdlExceptionWithValue<Type>
    {
        public UnknownEntityException() { }
        public UnknownEntityException(string errorCode, Type type, string message) : this(errorCode, type, message, (Exception)null) { }
        public UnknownEntityException(string errorCode, Type type, string message, params object[] args) : this(errorCode, type, string.Format(message, args), (Exception)null) { }
        public UnknownEntityException(string errorCode, Type type, string message, Exception inner) : base(errorCode, type, message, inner) { }
        protected UnknownEntityException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        public override string Message
        {
            get
            {
                return string.Format(@"{0} をエンティティとして解釈中: {1}".Ja(), Value, base.Message);
            }
        }
    }

    /// <summary>
    /// 不正なエンティティ記述を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public abstract class InvalidEntityException : InvalidSfdlExceptionWithValue<Type>
    {
        public InvalidEntityException() { }
        public InvalidEntityException(string errorCode, Type type, string message) : this(errorCode, type, message, (Exception)null) { }
        public InvalidEntityException(string errorCode, Type type, string message, params object[] args) : this(errorCode, type, string.Format(message, args), (Exception)null) { }
        public InvalidEntityException(string errorCode, Type type, string message, Exception inner) : base(errorCode, type, message, inner) { }
        protected InvalidEntityException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        public override string Message
        {
            get
            {
                return string.Format(@"{0} を{1}として解釈中: {2}".Ja(), Value, ElementName, base.Message);
            }
        }

        protected abstract FriendlyString ElementName { get; }
    }

    /// <summary>
    /// 不正なインターフェイス記述を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public class InvalidInterfaceException : InvalidEntityException
    {
        public InvalidInterfaceException() { }
        public InvalidInterfaceException(string errorCode, Type type, string message) : this(errorCode, type, message, (Exception)null) { }
        public InvalidInterfaceException(string errorCode, Type type, string message, params object[] args) : this(errorCode, type, string.Format(message, args), (Exception)null) { }
        public InvalidInterfaceException(string errorCode, Type type, string message, Exception inner) : base(errorCode, type, message, inner) { }
        protected InvalidInterfaceException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        protected override FriendlyString ElementName
        {
            get
            {
                return @"サービスインターフェイス".Ja();
            }
        }
    }

    /// <summary>
    /// 不正な構造体記述を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public class InvalidStructException : InvalidEntityException
    {
        public InvalidStructException() { }
        public InvalidStructException(string errorCode, Type type, string message) : this(errorCode, type, message, (Exception)null) { }
        public InvalidStructException(string errorCode, Type type, string message, params object[] args) : this(errorCode, type, string.Format(message, args), (Exception)null) { }
        public InvalidStructException(string errorCode, Type type, string message, Exception inner) : base(errorCode, type, message, inner) { }
        protected InvalidStructException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        protected override FriendlyString ElementName
        {
            get
            {
                return @"構造体".Ja();
            }
        }
    }

    /// <summary>
    /// 不正なインターフェイスメソッド記述を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public class InvalidMethodException : InvalidSfdlExceptionWithValue<MethodInfo>
    {
        public InvalidMethodException() { }
        public InvalidMethodException(string errorCode, MethodInfo info, string message) : base(errorCode, info, message) { }
        public InvalidMethodException(string errorCode, MethodInfo info, string message, params object[] args) : base(errorCode, info, string.Format(message, args)) { }
        public InvalidMethodException(string errorCode, MethodInfo info, string message, Exception inner) : base(errorCode, info, message, inner) { }
        protected InvalidMethodException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        public override string Message
        {
            get
            {
                return string.Format(@"{0}.{1} をメソッドとして解釈中: {2}".Ja(),
                    Value.DeclaringType, Value.Name, base.Message);
            }
        }
    }

    /// <summary>
    /// 不正なインターフェイスメソッドパラメータ記述を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public class InvalidMethodParameterException : InvalidSfdlExceptionWithValue<ParameterInfo>
    {
        public InvalidMethodParameterException() { }
        public InvalidMethodParameterException(string errorCode, ParameterInfo info, string message) : base(errorCode, info, message) { }
        public InvalidMethodParameterException(string errorCode, ParameterInfo info, string message, params object[] args) : base(errorCode, info, string.Format(message, args)) { }
        public InvalidMethodParameterException(string errorCode, ParameterInfo info, string message, Exception inner) : base(errorCode, info, message, inner) { }
        protected InvalidMethodParameterException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        public override string Message
        {
            get
            {
                return string.Format(@"{0}.{1} 中の {2} をパラメータとして解釈中: {3}".Ja(),
                    Value.Member.DeclaringType, Value.Member.Name, Value.Name, base.Message);
            }
        }
    }

    /// <summary>
    /// 不正な構造体フィールド記述を検知したときに投げられる共通例外型
    /// </summary>
    [Serializable]
    public class InvalidFieldException : InvalidSfdlExceptionWithValue<FieldInfo>
    {
        public InvalidFieldException() { }
        public InvalidFieldException(string errorCode, FieldInfo info, string message) : base(errorCode, info, message) { }
        public InvalidFieldException(string errorCode, FieldInfo info, string message, params object[] args) : base(errorCode, info, string.Format(message, args)) { }
        public InvalidFieldException(string errorCode, FieldInfo info, string message, Exception inner) : base(errorCode, info, message, inner) { }
        protected InvalidFieldException(SerializationInfo info, StreamingContext context) : base(info, context) { }

        public override string Message
        {
            get
            {
                return string.Format(@"{0}.{1} をメンバフィールドとして解釈中: {2}".Ja(),
                    Value.DeclaringType, Value.Name, base.Message);
            }
        }
    }
}
