﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    using Localization;

    public class StructCppCodeEmitter : EntityCppCodeEmitterBase<SfStruct>
    {
        internal StructCppCodeEmitter(SfStruct entity)
            : base(entity)
        {
        }

        public int GetStructSize()
        {
            return Entity.Size;
        }

        public override string GetCppTypeString(bool friendly)
        {
            var s = string.Format(@"{0}", GetMyCppName().FullName);
            return friendly ? s : @"::" + s;
        }

        public override string GetParameterString(string variableName, InOutType inOutType, bool friendly)
        {
            switch (inOutType)
            {
                case InOutType.Out:
                {
                    return base.GetParameterString(variableName, InOutType.Out, friendly);
                }
                case InOutType.In:
                {
                    variableName = variableName == null ? string.Empty : " " + variableName;
                    // TORIAEZU:
                    if (GetStructSize() > 8)
                    {
                        // サイズが 8 を超えるときは const & とする
                        return string.Format(@"const {0}&{1}", GetCppTypeString(friendly), variableName);
                    }
                    else
                    {
                        // サイズが 8 以下のときは値渡しとする
                        return string.Format(@"{0}{1}", GetCppTypeString(friendly), variableName);
                    }
                }
                default:
                {
                    throw new WrongImplementationException();
                }
            }
        }

        public override string GetStructPlacedString(string variableName, bool friendly)
        {
            return base.GetParameterString(variableName, InOutType.In, friendly);
        }

        public override void EmitForwardDeclarationCode(RawCppCodeGenerator rawGen)
        {
            ChangeMyNameSpace(rawGen);
            rawGen.WriteLine(@"struct {0};", GetMyCppName().GetSingleName());
        }

        public override void EmitDefinitionCode(RawCppCodeGenerator rawGen)
        {
            if (ExternalStructAttribute.GetSizeAndAlignment(Entity.InnerType) != null)
            {
                var message = string.Format(@"{0} で修飾される構造体型 {1} を出力することはできません。".Ja());
                throw new UserErrorException(@"SF-CPP-NotSupported:ExternalStruct", message, null);
            }
            var cppName = GetMyCppName();
            var name = cppName.GetSingleName();
            var fields = Entity.GetFields();
            ChangeMyNameSpace(rawGen);
            rawGen.WriteLine(@"struct {0}", name);
            rawGen.WriteLine(@"{");
            using (rawGen.Indent())
            {
                foreach (var field in fields)
                {
                    var fieldType = field.FieldType;
                    var fieldName = field.Name;
                    if (fieldType.IsArray)
                    {
                        fieldType = fieldType.GetElementType();
                        var length = FixedArrayAttribute.GetLength(field).Value;
                        fieldName = string.Format(@"{0}[{1}]", fieldName, length);
                    }
                    rawGen.WriteLine(@"{0};", fieldType.GetSfEntity().GetStructPlacedString(fieldName, true));
                }
            }
            rawGen.WriteLine(@"};");
        }

        public override void EmitReferenceCheckCode(RawCppCodeGenerator rawGen)
        {
            // TODO:
            // 型のサイズや、そのフィールドのオフセットが、
            // SFDL 処理系が想定しているものと一致しているかどうかの static_assert を出力する。
        }
    }
}
