﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;
using System.IO;
using Nintendo.ServiceFramework.CppCode;

namespace Nintendo.ServiceFramework.CppCode
{
    /// <summary>
    /// 指定した TextWriter に C++ のコードをいい感じに成形して書き出すためのクラス
    /// </summary>
    /// <remarks>
    /// 要素間の改行等は自動でいい感じに挿入される。
    /// </remarks>
    public class RawCppCodeGenerator
    {
        private static readonly CppName EmptySpace = new CppName();

        private TextWriter m_Writer;

        /// <summary>
        /// 現在の名前空間を表す
        /// </summary>
        /// <remarks>
        /// 変更には ChangeCurrentNamespace を使用する。
        /// </remarks>
        public CppName CurrentNamespace { get; private set; }

        /// <summary>
        /// NOLINT を出力するかどうか。
        /// </summary>
        public bool IsNoLintEnabled { get; set; }

        private bool m_NeedsAutoNewline = false;

        /// <summary>
        /// 指定した TextWriter に出力するインスタンスを作成する。
        /// </summary>
        /// <param name="writer">出力先の TextWriter</param>
        public RawCppCodeGenerator(TextWriter writer)
        {
            this.m_Writer = writer;
            this.CurrentNamespace = EmptySpace;
            this.IsNoLintEnabled = false;
        }

        /// <summary>
        /// 文字列を出力し改行する
        /// </summary>
        /// <param name="s">出力文字列</param>
        public void WriteLine(string s)
        {
            if (m_NeedsAutoNewline)
            {
                m_Writer.WriteLine();
                this.m_NeedsAutoNewline = false;
            }
            m_Writer.WriteLine((new string(' ', IndentLevel) + s).TrimEnd());
            this.m_PreviousIsInclude = false;
        }

        /// <summary>
        /// フォーマット付きで文字列を出力し改行する
        /// </summary>
        /// <param name="format">出力するフォーマット文字列</param>
        /// <param name="args">フォーマット引数</param>
        public void WriteLine(string format, params string[] args)
        {
            WriteLine(string.Format(format, args));
        }

        /// <summary>
        /// 出力を完了する
        /// </summary>
        public void Finish()
        {
            CloseCurrentNamespace();
            m_Writer.Flush();
        }

        #region インデント管理

        private int IndentLevel { get; set; }

        public class Indenter : IDisposable
        {
            private RawCppCodeGenerator m_RawCppCodeGenerator;
            private int m_IndentLevel;

            internal Indenter(RawCppCodeGenerator rawCppCodeGenerator, int indentLevel)
            {
                this.m_RawCppCodeGenerator = rawCppCodeGenerator;
                this.m_IndentLevel = indentLevel;
                rawCppCodeGenerator.IndentLevel += indentLevel;
            }

            public void Dispose()
            {
                m_RawCppCodeGenerator.IndentLevel -= m_IndentLevel;
            }
        }

        public Indenter Indent(int indentLevel = 4)
        {
            return new Indenter(this, indentLevel);
        }

        #endregion

        #region namespace コントロール

        /// <summary>
        /// 明示的に名前空間を変更する
        /// </summary>
        /// <param name="space">名前空間名</param>
        public void ChangeCurrentNamespace(CppName space)
        {
            if (CurrentNamespace == space)
            {
                return;
            }
            CloseCurrentNamespace();
            OpenNamespace(space);
        }

        private void OpenNamespace(CppName space)
        {
            var s = string.Join(" ", space.NameChain.Select(n => string.Format(@"namespace {0} {{", n)).ToArray());
            if (IsNoLintEnabled)
            {
                s += @" // NOLINT(whitespace/braces)";
            }
            WriteLine(s);
            this.CurrentNamespace = space;
            this.m_NeedsAutoNewline = true;
        }

        /// <summary>
        /// 名前空間を閉じる
        /// </summary>
        public void CloseCurrentNamespace()
        {
            if (CurrentNamespace == EmptySpace)
            {
                return;
            }
            WriteLine(new string('}', CurrentNamespace.Rank));
            CurrentNamespace = EmptySpace;
            this.m_NeedsAutoNewline = true;
        }

        #endregion

        #region Newline

        /// <summary>
        /// 強制的に改行する
        /// </summary>
        public void Newline()
        {
            Newline(1);
        }

        /// <summary>
        /// 指定した回数だけ、強制的に改行する
        /// </summary>
        /// <param name="n">改行回数</param>
        public void Newline(int n)
        {
            for (var i = 0; i < n; ++i)
            {
                m_Writer.WriteLine();
            }
            this.m_NeedsAutoNewline = false;
            this.m_PreviousIsInclude = false;
        }

        /// <summary>
        /// 次の要素を書き出す直前に改行するように指示する
        /// </summary>
        public void AutoNewLine()
        {
            this.m_NeedsAutoNewline = true;
        }

        #endregion

        #region Include

        private bool m_PreviousIsInclude = false;

        private HashSet<string> m_IncludedPaths = new HashSet<string>();

        /// <summary>
        /// s を #include する文字列を出力する
        /// </summary>
        /// <param name="s">#include の引数</param>
        /// <param name="closeNamespace">true の場合、#include の前で名前空間を閉じる</param>
        /// <remarks>既に #include したことのあるものが指定された場合には、何もしない。</remarks>
        public void Include(string s, bool closeNamespace = true)
        {
            if (m_IncludedPaths.Contains(s))
            {
                return;
            }
            IncludeImpl(s, closeNamespace);
            m_IncludedPaths.Add(s);
        }

        /// <summary>
        /// 強制的に s を #include する文字列を出力する
        /// </summary>
        /// <param name="s">#include の引数</param>
        /// <remarks>他の実装ファイルなどを強制的に #include する場合に用いる。</remarks>
        public void ForceInclude(string s)
        {
            IncludeImpl(s, false);
        }

        private void IncludeImpl(string s, bool closeNamespace)
        {
            if (closeNamespace)
            {
                CloseCurrentNamespace();
            }
            if (m_PreviousIsInclude)
            {
                this.m_NeedsAutoNewline = false;
            }
            WriteLine(@"#include {0}", s);
            this.m_NeedsAutoNewline = true;
            this.m_PreviousIsInclude = true;
        }

        #endregion
    }
}
