﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    public abstract class EntityCppCodeEmitter
    {
        public SfEntity Entity { get; private set; }

        protected EntityCppCodeEmitter(SfEntity entity)
        {
            this.Entity = entity;
        }

        public virtual string GetForwardString(string variableName, InOutType inOutType, bool friendly)
        {
            return variableName;
        }

        /// <summary>
        /// このエンティティが C++ 上で関数引数となったときの文字列を取得する
        /// </summary>
        /// <param name="variableName">宣言名</param>
        /// <param name="inOutType">in/out 種別</param>
        /// <param name="friendly">ユーザのよみやすさを優先した記述をするかどうか</param>
        /// <returns></returns>
        public virtual string GetParameterString(string variableName, InOutType inOutType, bool friendly)
        {
            variableName = variableName == null ? string.Empty : " " + variableName;
            switch (inOutType)
            {
                case InOutType.Out:
                {
                    var nnSfOut = @"nn::sf::Out";
                    return string.Format(@"{0}<{1}>{2}", friendly ? nnSfOut : @"::" + nnSfOut, GetCppTypeString(friendly), variableName);
                }
                case InOutType.In:
                {
                    return string.Format(@"{0}{1}", GetCppTypeString(friendly), variableName);
                }
                default:
                {
                    throw new WrongImplementationException();
                }
            }
        }

        /// <summary>
        /// このエンティティが C++ 上で型として現れる際(テンプレート引数や返り値)の文字列を取得する
        /// </summary>
        /// <param name="friendly">ユーザのよみやすさを優先した記述をするかどうか</param>
        /// <returns></returns>
        public abstract string GetCppTypeString(bool friendly);

        /// <summary>
        /// このエンティティが C++ の構造体に配置される際の文字列を取得する
        /// </summary>
        /// <param name="variableName">宣言名</param>
        /// <param name="friendly">ユーザのよみやすさを優先した記述をするかどうか</param>
        /// <returns></returns>
        /// <exception cref="WrongImplementationException">構造体に配置できないエンティティである</exception>
        public virtual string GetStructPlacedString(string variableName, bool friendly)
        {
            throw new WrongImplementationException();
        }

        /// <summary>
        /// このエンティティが外部で定義されてる際に参照するために必要なヘッダパス列挙を取得する
        /// </summary>
        /// <returns>ヘッダパス列挙</returns>
        public virtual IEnumerable<string> GetExternalIncludes()
        {
            return CppRefPathAttribute.GetCppRefPaths(Entity);
        }

        /// <summary>
        /// このエンティティを定義するために必要なヘッダパスを取得する
        /// </summary>
        /// <returns>ヘッダパス列挙</returns>
        public virtual IEnumerable<string> GetAdditionalIncludes()
        {
            yield break;
        }

        /// <summary>
        /// 前方宣言用の文字列列挙を出力する
        /// </summary>
        /// <param name="rawGen">出力対象の低レイヤジェネレータ</param>
        public abstract void EmitForwardDeclarationCode(RawCppCodeGenerator rawGen);

        /// <summary>
        /// 定義用の文字列列挙を取得する
        /// </summary>
        /// <param name="rawGen">出力対象の低レイヤジェネレータ</param>
        public abstract void EmitDefinitionCode(RawCppCodeGenerator rawGen);

        /// <summary>
        /// 参照チェック用の文字列列挙を取得する
        /// </summary>
        /// <param name="rawGen">出力対象の低レイヤジェネレータ</param>
        public virtual void EmitReferenceCheckCode(RawCppCodeGenerator rawGen)
        {
            // nop
        }

        #region 派生クラス実装用ユーティリティ

        /// <summary>
        /// エンティティの CppName を取得する
        /// </summary>
        /// <returns></returns>
        internal CppName GetMyCppName()
        {
            return new CppName(CppFullNameAttribute.GetCppFullName(Entity.InnerType));
        }

        /// <summary>
        /// エンティティの宣言名前空間にジェネレータを設定する
        /// </summary>
        /// <param name="rawGen">出力対象の低レイヤジェネレータ</param>
        internal void ChangeMyNameSpace(RawCppCodeGenerator rawGen)
        {
            rawGen.ChangeCurrentNamespace(GetMyCppName().GetNameSpaceName());
        }

        #endregion
    }

    public abstract class EntityCppCodeEmitterBase<T> : EntityCppCodeEmitter
        where T : SfEntity
    {
        protected EntityCppCodeEmitterBase(T entity)
            : base(entity)
        {
        }

        public new T Entity
        {
            get
            {
                return (T)base.Entity;
            }
        }
    }
}
