﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    /// <summary>
    /// C++ のシンボル名を表すためのクラス
    /// </summary>
    public class CppName
    {
        private IEnumerable<string> m_Names;

        public CppName(params string[] args)
            : this(args.SelectMany(SplitName).ToArray().AsEnumerable())
        {
        }

        public static CppName Create(params string[] args)
        {
            return new CppName(args);
        }

        private CppName(IEnumerable<string> e)
        {
            this.m_Names = e;
        }

        internal static CppName CreateDirect(IEnumerable<string> e)
        {
            return new CppName(e);
        }

        private static readonly string[] SplitterArray = new[] { @"::" };
        private static string[] SplitName(string fullName)
        {
            if (fullName.Length == 0)
            {
                return new string[0];
            }
            return fullName.Split(SplitterArray, StringSplitOptions.None);
        }

        public string FullName
        {
            get
            {
                return JoinNames(m_Names);
            }
        }

        public int Rank
        {
            get
            {
                return m_Names.Count();
            }
        }

        private static string JoinNames(IEnumerable<string> e)
        {
            return string.Join(@"::", e);
        }

        public IEnumerable<string> NameChain
        {
            get
            {
                return m_Names;
            }
        }

        public string GetRelativeName(CppName current)
        {
            var n = 0;
            foreach (var e in m_Names.Zip(current.m_Names, Tuple.Create))
            {
                if (!(e.Item1 == e.Item2))
                {
                    break;
                }
                ++n;
            }
            return JoinNames(m_Names.Skip(n));
        }

        public CppName GetNameSpaceName()
        {
            var a = m_Names.ToArray();
            if (!(a.Length > 0))
            {
                throw new ArgumentException();
            }
            return new CppName(a.Take(a.Length - 1));
        }

        public string GetSingleName()
        {
            return m_Names.Last();
        }

        public static CppName operator +(CppName lhs, CppName rhs)
        {
            return new CppName(lhs.m_Names.Concat(rhs.m_Names));
        }

        public static CppName operator +(CppName lhs, string s)
        {
            return new CppName(lhs.m_Names.Concat(SplitName(s)));
        }

        #region equality

        public static bool operator ==(CppName lhs, CppName rhs)
        {
            var rhsIsNull = object.ReferenceEquals(rhs, null);
            if (object.ReferenceEquals(lhs, null))
            {
                return rhsIsNull;
            }
            else
            {
                return !rhsIsNull && lhs.FullName == rhs.FullName;
            }
        }

        public static bool operator !=(CppName lhs, CppName rhs)
        {
            return !(lhs == rhs);
        }

        public override bool Equals(object obj)
        {
            var o = obj as CppName;
            return (object)o != null && this == o;
        }

        public override int GetHashCode()
        {
            return this.FullName.GetHashCode();
        }

        #endregion

        public override string ToString()
        {
            return this.FullName;
        }
    }
}
