﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    public class CppCodeGenerator
    {
        /// <summary>
        /// 指定されたエンティティを書き出す
        /// </summary>
        /// <param name="rawGen">低位コードジェネレータ</param>
        /// <param name="definees">書き出すエンティティ配列</param>
        /// <remarks>
        /// 権利ヘッダや #pragma once の書き出しは行わない。
        /// 書き出し後に rawGen.Finish() の呼び出しは行わないため、必要に応じて手動で呼ぶ必要がある。
        /// </remarks>
        public void Generate(RawCppCodeGenerator rawGen, params SfEntity[] definees)
        {
            var allRefs = CollectReferredEntities(definees);
            var externalRefs = GetExternalRefs(allRefs, definees);
            // 参照するが外部で定義されているものに関してヘッダをインクルード
            GenerateExternalIncludes(rawGen, externalRefs);
            // ここで定義するために必要なヘッダをインクルード
            GenerateAdditionalIncludes(rawGen, definees);
            rawGen.AutoNewLine();
            // ここで定義するものを先行宣言
            GenerateForwardDeclarations(rawGen, definees);
            rawGen.AutoNewLine();
            // 定義
            GenerateDefinitions(rawGen, definees);
            rawGen.AutoNewLine();
            // この定義内で参照しているものすべてに関して、静的なチェック
            GenerateReferenceChecks(rawGen, allRefs);
        }

        private static HashSet<SfEntity> CollectReferredEntities(SfEntity[] entities)
        {
            var ret = new HashSet<SfEntity>(entities.SelectMany(e => e.GetReferredEntities()));
            foreach (var e in entities)
            {
                ret.Add(e);
            }
            return ret;
        }

        private static HashSet<SfEntity> GetExternalRefs(HashSet<SfEntity> allRefs, IEnumerable<SfEntity> definees)
        {
            var externalRefs = new HashSet<SfEntity>(allRefs);
            externalRefs.ExceptWith(definees);
            return externalRefs;
        }

        private static void GenerateExternalIncludes(RawCppCodeGenerator rawGen, IEnumerable<SfEntity> externalRefs)
        {
            var paths = from e in externalRefs
                        from path in e.GetExternalIncludes()
                        orderby path
                        select path;
            foreach (var path in paths)
            {
                rawGen.Include(path);
            }
        }

        private static void GenerateAdditionalIncludes(RawCppCodeGenerator rawGen, IEnumerable<SfEntity> definee)
        {
            // definee の CppRefPath は必要に応じて GenerateExternalIncludes されているはずなので
            // GenerateAdditionalIncludes しないようにする
            var except = from e in definee
                         from path in CppRefPathAttribute.GetCppRefPaths(e)
                         select path;
            var paths = from e in definee
                        from path in e.GetAdditionalIncludes()
                        where !except.Contains(path)
                        orderby path
                        select path;
            foreach (var path in paths)
            {
                rawGen.Include(path);
            }
        }

        // 比較に使用するためのキーを返す
        private static string GetSfEntityCompareKey(SfEntity e)
        {
            // C++ の完全名を利用する
            return CppFullNameAttribute.GetCppFullName(e.InnerType);
        }

        private static void GenerateForwardDeclarations(RawCppCodeGenerator rawGen, IEnumerable<SfEntity> definees)
        {
            foreach (var e in definees.OrderBy(GetSfEntityCompareKey))
            {
                e.EmitForwardDeclarationCode(rawGen);
                rawGen.AutoNewLine();
            }
        }

        private static void GenerateDefinitions(RawCppCodeGenerator rawGen, IEnumerable<SfEntity> definees)
        {
            var comparer = Comparer<SfEntity>.Create((x, y) => GetSfEntityCompareKey(x).CompareTo(GetSfEntityCompareKey(y)));
            var list = new SortedSet<SfEntity>(definees, comparer);
            while (!(list.Count == 0))
            {
                var e = PopSfEntityToBeDefined(list);
                e.EmitDefinitionCode(rawGen);
                rawGen.AutoNewLine();
            }
        }

        private static SfEntity PopSfEntityToBeDefined(SortedSet<SfEntity> list)
        {
            // 先頭にある要素に関して、
            var entity = list.First();
            while (true)
            {
                // 強い依存があり、まだ生成される必要のある(list.Contains)ものを抽出し、
                var strongRefs = from e in entity.GetStronglyReferredEntities()
                                 where list.Contains(e)
                                 orderby GetSfEntityCompareKey(e)
                                 select e;
                // その先頭の要素があれば取得し、
                var strongRef = strongRefs.FirstOrDefault();
                if (strongRef == null)
                {
                    // なければ、現在対象としているものを list から削除して返す。
                    list.Remove(entity);
                    return entity;
                }
                // あった場合には、その要素が強く依存する要素がさらにないかどうかを確かめるため、
                // 対象を変更してループを回す。
                entity = strongRef;
            }
        }

        private static void GenerateReferenceChecks(RawCppCodeGenerator rawGen, IEnumerable<SfEntity> allRefs)
        {
            foreach (var e in allRefs.OrderBy(GetSfEntityCompareKey))
            {
                e.EmitReferenceCheckCode(rawGen);
                rawGen.AutoNewLine();
            }
        }
    }
}
