﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Net;
using System.Globalization;
using System.Collections.Generic;
using Nintendo.ControlTarget;
using Nintendo.Bridge;

namespace Nintendo.UpdateHostBridge
{
    public enum ExitStatus
    {
        Success,
        Failure
    }

    public class ControlHostBridgeUtility : IDisposable
    {
        private bool m_IsPrintVersbose = false;
        private bool m_IsDisposed = false;

        public ControlHostBridgeUtility()
        {
        }

        ~ControlHostBridgeUtility()
        {
            Dispose();
        }

        public void Dispose()
        {
            if (!m_IsDisposed)
            {
                m_IsDisposed = true;
            }
        }

        public void SetVerboseFlag(bool flag)
        {
            m_IsPrintVersbose = flag;
        }

        public bool GetVerboseFlag()
        {
            return m_IsPrintVersbose;
        }

        private bool IsRecoveryUpdateRequiredHw(String hwVersion)
        {
            // SDEV 1.6-SDEV PreMP2
            return 2<= Convert.ToInt32(hwVersion) && Convert.ToInt32(hwVersion) <= 6;
        }

        public void PrintProgress(string message)
        {
            if( GetVerboseFlag() )
            {
                Console.Out.WriteLine(message);
            }
            else
            {
                Console.Out.Write(".");
            }
        }

        public FileInfo GetUpdateFirmwarePath(bool recovery)
        {
            var root = ControlTarget.PathUtility.FindSigloRoot();

            if (recovery)
            {
                var fwPath = new FileInfo(Path.Combine(root, @"Resources\Firmwares\Nx\HostBridgeRecoveryFirmware.nhf"));
                if (!fwPath.Exists)
                {
                    fwPath = new FileInfo(Path.Combine(root, @"Externals\HostBridge\images\recovery.nhf"));
                }
                return fwPath;
            }
            else
            {
                var fwPath = new FileInfo(Path.Combine(root, @"Resources\Firmwares\Nx\HostBridgeFirmware.nhf"));
                if (!fwPath.Exists)
                {
                    fwPath = new FileInfo(Path.Combine(root, @"Externals\HostBridge\images\package.nhf"));
                }
                return fwPath;
            }
        }

         // IPアドレスが渡されたらTargetInformationFromTargetManager.ipAddress以外は空の状態で返します
        public static Utility.TargetInformationFromTargetManager GetIPOrFindSingleTarget(string target, bool printErrorIfFailed, bool verbose)
        {
            var targetinfo = new Utility.TargetInformationFromTargetManager();
            if(Utility.IsIPAddress(target))
            {
                targetinfo.ipAddress = target;
                return targetinfo;
            }
            else
            {
                targetinfo = Utility.FindSingleTarget(target, printErrorIfFailed, verbose);
                return targetinfo;
            }
        }

        public ExitStatus Update(UpdateArgument args)
        {
            var targetInfo = GetIPOrFindSingleTarget(args.Target, true, args.IsVerbose);
            if (targetInfo == null)
            {
                return ExitStatus.Failure;
            }
            PrintProgress("FindSingleTarget Complete.");

            FileInfo fwPath = args.FirmwareImagePath;
            if (fwPath == null)
            {
                fwPath = GetUpdateFirmwarePath(args.IsRecovery);
            }
            if (!fwPath.Exists)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("\"" + fwPath.FullName + "\" is not found");
                return ExitStatus.Failure;
            }
            PrintProgress("GetUpdateFirmwarePath Complete.");

            if (GetVerboseFlag())
            {
                Console.Out.WriteLine("Firmware path: " + fwPath.FullName);
            }

            PrintProgress("CheckNeedUpdate Complete.");

            BridgeUpdate update = new BridgeUpdate();
            update.Verbose = args.IsVerbose;
            if (args.IsRecovery)
            {
                if (!update.UpdateBridge(args.Target, fwPath.FullName, true))
                {
                    Console.Error.WriteLine("failed.");
                    return ExitStatus.Failure;
                }
            }
            else
            {
                if (!update.UpdateBridge(args.Target, fwPath.FullName))
                {
                    Console.Error.WriteLine("failed.");
                    return ExitStatus.Failure;
                }
            }

            PrintProgress("UpdateFirmware Complete.");

            if (args.IsRecovery)
            {
                return ExitStatus.Success;
            }

            PrintProgress("HostBridge has been rebooted.");

            if (args.IsWaitReboot)
            {
                Console.Out.Write("Waiting for reboot.");

                // reboot コマンドにより実際に再起動中になって検出できなくなるまで待つ
                Thread.Sleep(TimeSpan.FromSeconds(20));

                const int MaxRetryCount = 12;
                int count = 0;
                using (var accessor = new TargetManagerAccessor())
                {
                    // InitializeSdev から呼び出される場合は TM は起動済みである想定
                    accessor.SetStartTrayIcon(true);
                    accessor.EnsureStart();

                    while (count < MaxRetryCount)
                    {
                        var targets = accessor.ListDetectedTarget(TimeSpan.FromSeconds(5));

                        foreach (var target in targets)
                        {
                            if (Utility.IsIPAddress(args.Target))
                            {
                                if (target.GetIpAddress() == args.Target)
                                {
                                    goto detected;
                                }
                            }
                            else
                            {
                                if (target.GetMacAddress() == args.Target)
                                {
                                    goto detected;
                                }
                            }
                        }

                        count++;
                        Console.Out.Write(".");
                    }
                }
            detected:
                if (count >= MaxRetryCount)
                {
                    Console.Error.WriteLine("Timeout.");
                    return ExitStatus.Failure;
                }
            }

            return ExitStatus.Success;
        }

        public ExitStatus IsRecoveryUpdateRequired(string target, out bool required)
        {
            required = false;
            var targetInfo = GetIPOrFindSingleTarget(target, true, false);
            if (targetInfo == null)
            {
                return ExitStatus.Failure;
            }
            PrintProgress("FindSingleTarget Complete.");

            BridgeInfo info = new BridgeInfo();
            string sion;
            string tics;
            info.GetRecoveryHash(target, out sion, out tics);

            if ((sion == "2d9130c447838cabfb04ad9e480704df17fe55dd") ||
                (tics == "8e185403b26c49d0c71f1a83da41ed2290802ca6"))
            {
                return ExitStatus.Success;
            }

            string hwVersion;
            if (info.GetHwVersion(target, out hwVersion))
            {
                if (!IsRecoveryUpdateRequiredHw(hwVersion))
                {
                    return ExitStatus.Success;
                }
            }
            else
            {
                return ExitStatus.Failure;
            }

            PrintProgress("UpdateFirmware Complete.");

            return ExitStatus.Success;
        }
    }
}
