﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using Nintendo.Foundation.IO;
using Nintendo.ControlTarget;

namespace Nintendo.UpdateHostBridge
{
    public class CommandInterface
    {
        public enum ExitCode
        {
            Success = 0,
            RecoveryUpdateRequired = 8,
            RecoveryUpdateNotRequired = 9
        }

        public static int Main(string[] rawArgs)
        {
            try
            {
                NnRunCommandArgumentRaw parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<NnRunCommandArgumentRaw>(rawArgs, out parsed))
                {
                    return 1;
                }
                else if (parsed.List != null)
                {
                    return List();
                }
                else if (parsed.Update != null)
                {
                    Console.Out.Write("Updating.");
#if DEBUG
                    Console.Out.WriteLine();
#endif
                    var args = new UpdateArgument(parsed.Update);
                    int ret = Update(args);
                    if (ret == 0)
                    {
                        Console.Out.WriteLine("Complete.");
                    }
                    return ret;
                }
                else if (parsed.IsRecoveryUpdateRequired != null)
                {
                    bool required;
                    int ret = IsRecoveryUpdateRequired(parsed.IsRecoveryUpdateRequired.Target, out required);
                    if (ret == 0)
                    {
                        if (required)
                        {
                            Console.Out.WriteLine("IsRecoveryUpdateRequired=True");
                            ret = (int)ExitCode.RecoveryUpdateRequired;
                        }
                        else
                        {
                            Console.Out.WriteLine("IsRecoveryUpdateRequired=False");
                            ret = (int)ExitCode.RecoveryUpdateNotRequired;
                        }
                    }
                    return ret;
                }
                else
                {
                    throw new Exception("Unknown command");
                }
            }
            catch (Exception exception)
            {
                PrintException(exception);
                return 1;
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Out.WriteLine();
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.Error.WriteLine("StackTrace: {0}", exception.StackTrace);

            Trace.WriteLine("StackTrace: {0}", exception.StackTrace);
        }

        public static int List()
        {
            return Utility.ListDetectedTarget() == ControlTarget.ExitStatus.Success ? 0 : 1;
        }

        public static int Update(UpdateArgument args)
        {
            using (var handle = new ControlHostBridgeUtility())
            {
                handle.SetVerboseFlag(args.IsVerbose);
                ExitStatus status = handle.Update(args);
                if (status != ExitStatus.Success)
                {
                    return handle.Update(args) == ExitStatus.Success ? 0 : 1;
                }
                else
                {
                    return 0;
                }
            }
        }

        public static int IsRecoveryUpdateRequired(string target, out bool required)
        {
            using (var handle = new ControlHostBridgeUtility())
            {
                return handle.IsRecoveryUpdateRequired(target, out required) == ExitStatus.Success ? 0 : 1;
            }
        }

    }
}
