﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using System.Globalization;
using System.Text.RegularExpressions;
using CommandUtility;

namespace RunSystemInitializerManu
{
    public class RunSystemInitializerManuArgument
    {
        [CommandLineOption('i', "image",
            Description = "set initial image.", IsRequired = true)]
        public string ImagePath { get; set; }

        [CommandLineOption("initializer",
            Description = "set initializer id.", IsRequired = true)]
        public string InitializerId { get; set; }

        [CommandLineOption("tmpdir",
            Description = "set temporary directory.", IsRequired = true)]
        public string TemporaryDirectory { get; set; }

        [CommandLineOption("config",
            Description = "set config file.", IsRequired = true)]
        public string Config { get; set; }

        [CommandLineOption("timeout",
            Description = "set timeout(seconds).", DefaultValue = 300)]
        public int Timeout { get; set; }

        [CommandLineOption('v', "verbose",
            Description = "Output trace logs.")]
        public bool Verbose { get; set; }

        [CommandLineOption('u', "usb_device_path",
            Description = "USB Device path.")]
        public string UsbDevicePath { get; set; }

        [CommandLineValues(Description = "set append argument")]
        public string[] Arguments { get; set; }

        private string RequestPath;
        private string ResponsePath;
        private string ResultPath;

        public int Run()
        {
            TemporaryDirectory = Path.Combine(TemporaryDirectory, RemoveInvalidFileNameChars(UsbDevicePath));
            Directory.CreateDirectory(TemporaryDirectory);

            RequestPath = Path.GetFullPath(Path.Combine(TemporaryDirectory, "request.bin"));
            ResponsePath = Path.GetFullPath(Path.Combine(TemporaryDirectory, "response.bin"));
            ResultPath = Path.GetFullPath(Path.Combine(TemporaryDirectory, "result.txt"));

            if (File.Exists(RequestPath))
            {
                File.Delete(RequestPath);
            }

            if (File.Exists(ResponsePath))
            {
                File.Delete(ResponsePath);
            }

            if (File.Exists(ResultPath))
            {
                File.Delete(ResultPath);
            }

            CancellationTokenSource cancelSource = new CancellationTokenSource();

            try
            {
                var initializeTask = Task.Run(() => RunSystemInitializerManu(cancelSource));
                var requestTask = Task.Run(() => RequestWsIssuer(Config, cancelSource));

                Task.WaitAll(new Task[] { initializeTask, requestTask }, (int)TimeSpan.FromSeconds(Timeout).TotalMilliseconds, cancelSource.Token);
            }
            finally
            {
                if (cancelSource.IsCancellationRequested)
                {
                    ProcessUtility.KillProcessBasedOnCommandLineArgument("RunOnTargetFromNand.exe", UsbDevicePath);
                    ProcessUtility.KillProcessBasedOnCommandLineArgument("RequestToWsIssuer.exe", UsbDevicePath);
                }
                cancelSource.Token.ThrowIfCancellationRequested();
            }

            CheckResultFile();

            return 0;
        }

        private void RequestWsIssuer(string Config, CancellationTokenSource cancel)
        {
            try
            {
                for (int i = 0; i < Timeout; i++)
                {
                    if (File.Exists(RequestPath) && new FileInfo(RequestPath).Length == 256)
                    {
                        break;
                    }

                    Thread.Sleep(TimeSpan.FromSeconds(1));

                    Console.WriteLine("Wait {0}/{1}, path = {2}", i + 1, Timeout, RequestPath);
                }

                CommandUtility.RetryUtility.Do(() =>
                {
                    SdkTool.Execute("RequestToWsIssuer.exe",
                    "--input",
                    RequestPath,
                    "--output",
                    ResponsePath,
                    "--config",
                    Path.GetFullPath(Config));

                },
                (e) => {
                     Console.WriteLine("RequestWsIssuer: {0}", e.Message);
                },
                Timeout,
                TimeSpan.FromSeconds(1));
            }
            catch (Exception e)
            {
                Console.WriteLine("Error in RequestWsIssuer: {0}", e.Message);
                cancel.Cancel();
                throw;
            }
        }

        private void RunSystemInitializerManu(CancellationTokenSource cancel)
        {
            try
            {
                SdkTool.Execute("RunOnTargetFromNand.exe",
                    "--usb_device_path", String.Format("{0}",UsbDevicePath),
                    "--id", InitializerId,
                    "--timeout", String.Format("{0}", Timeout),
                    "--args", SdkTool.MakeArgumentString(new string[] {
                    "-i",
                    Path.GetFullPath(ImagePath),
                    "--output-request",
                    RequestPath,
                    "--input-response",
                    ResponsePath,
                    "--output-result",
                    ResultPath
                }.Concat(this.Arguments == null ? new string[] { } : this.Arguments).ToArray()));
            }
            catch (Exception e)
            {
                Console.WriteLine("Error in RunSystemInitializerManu: {0}", e.Message);
                Console.WriteLine("Error in RunSystemInitializerManu: {0}", e.StackTrace);
                cancel.Cancel();
                throw;
            }
        }

        private void CheckResultFile()
        {
            if (!File.Exists(ResultPath))
            {
                throw new Exception(string.Format("Not found result file: {0}", ResultPath));
            }

            var resultText = File.ReadAllText(ResultPath);
            Console.WriteLine("ResultFromTarget: {0}", resultText);

            if (resultText.Trim() != "Success")
            {
                throw new Exception(string.Format("Failed to initialize target: {0}", resultText));
            }
        }

        private string RemoveInvalidFileNameChars(string str)
        {
            if (String.IsNullOrEmpty(str))
            {
                return String.Empty;
            }
            var removeChars = Path.GetInvalidFileNameChars();
            var removeCharsString = new string(removeChars);
            var removePattern = string.Format("[{0}]", Regex.Escape(removeCharsString));
            var removedStr = Regex.Replace(str, removePattern, "");

            return removedStr;
        }
    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
                RunSystemInitializerManuArgument parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<RunSystemInitializerManuArgument>(args, out parsed))
                {
                    System.Environment.Exit(1);
                }

                Log.Verbose = true;

                System.Environment.Exit(parsed.Run());
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.Error.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
        }
    }
}
