﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Net;
using Nintendo.ControlTarget;
using Nintendo.ControlTarget.ConsoleShell;

namespace Nintendo.RunOnTarget
{
    public class RunnerPrivate
    {
        public CommonRunCommandArgument Arguments { get; private set; }
        public bool MonitorShell { get; set; }
        public TimeSpan WaitAfterReset { get; set; }

        public RunnerPrivate()
        {
        }

        internal ExitStatus Run(RunCommandArgumentPrivate args, string extraTargetManagerDirectory)
        {
            Arguments = args.Common;
            MonitorShell = args.MonitorShell;
            WaitAfterReset = TimeSpan.FromSeconds(args.WaitSecondsAfterReset);
            return Run(Arguments.ProgramOutput, Arguments.TargetOutput, Arguments.Program, Arguments.ExitCondition, args.TargetName, args.HostBridgeName, args.SuppressAutoKill, extraTargetManagerDirectory, Arguments.Arguments, Arguments.EnvDefinition, args.MonitorSerial, Arguments.Filter);
        }

        public ExitStatus Run(ITargetBlock<string> output, ITargetBlock<Tuple<string, string>> targetOutput, TargetProgramInfo program, ExitCondition exitCondition, string targetName, string hostBridgeName, bool suppressAutoKill, string extraTargetManagerDirectory, string[] arguments = null, FileInfo envDefinitionFile = null, bool monitorSerial = false, string filter = null)
        {
            ExitStatus exitStatus;
            var timeoutForExit = new TimeoutForExit();

            try
            {
                using (var targetManager = new TargetManagerAccessor())
                {
                    if (extraTargetManagerDirectory != null)
                    {
                        targetManager.SetExtraTargetManagerDirectory(extraTargetManagerDirectory);
                    }
                    targetManager.EnsureStart();

                    ISourceBlock<string> serialLogSource = new BufferBlock<string>();
                    HostBridgeAccessor serialAccessor = null;
                    if (monitorSerial)
                    {
                        var targetInfo = targetManager.FindTargetInfo(hostBridgeName);
                        if (targetInfo != null && targetInfo.GetConnectionType() == "Ethernet")
                        {
                            serialAccessor = new HostBridgeAccessor(NetworkUtility.Parse(targetInfo.GetIpAddress(), 10023));
                            serialLogSource = serialAccessor.ReadPort;
                        }
                        else
                        {
                            throw new Exception(string.Format("Found no target for serial output. name = {0}", hostBridgeName));
                        }
                    }
                    var serialLogDecorator = DataflowUtility.CreateAdditionMetaBlock("serial");

                    using (DataflowUtility.LinkBlock("serialLogSource->serialLogDecorator", serialLogSource, serialLogDecorator))
                    using (DataflowUtility.LinkBlock("serialLogDecorator->targetOutput", serialLogDecorator, targetOutput))
                    {

                        if (Arguments.Reset)
                        {
                            Trace.WriteLine("Reset the target.");
                            var target_ = targetManager.GetTarget(targetManager.FindTarget(hostBridgeName));
                            targetManager.RebootTarget(target_);
                            Task.Delay(WaitAfterReset).Wait();
                        }

                        if (Arguments.WorkingDirectory != null)
                        {
                            Trace.WriteLine($"Set target working directory. path='{Arguments.WorkingDirectory.FullName}'");
                            using (var target_ = targetManager.GetTarget(targetManager.FindTarget(targetName)))
                            {
                                target_.SetTargetWorkingDirectory(Arguments.WorkingDirectory.FullName);
                            }
                        }

                        var target = targetManager.FindTargetInfo(targetName);
                        var coodinator = new RunnerToolCoordinator(target.GetSerialNumber());

                        using (var targetOwnership = coodinator.RequestOwnership())
                        using (TargetCommunicationInterface ports = new BasicTargetCommunication(targetManager, targetName, filter, Arguments.ConnectTimeout))
                        {
                            var targetLogSource = ports.TargetLog;
                            var logMonitoringSource = new BufferBlock<string>();
                            var targetLogDecorator = DataflowUtility.CreateAdditionMetaBlock("target");

                            try
                            {
                                using (DataflowUtility.LinkBlock("targetLogSource->targetLogDecorator", targetLogSource, targetLogDecorator))
                                using (DataflowUtility.LinkBlock("targetLogSource->output", targetLogSource, output))
                                using (DataflowUtility.LinkBlock("targetLogDecorator->targetOutput", targetLogDecorator, targetOutput))

                                {
                                    if (!Arguments.Reset && !suppressAutoKill)
                                    {
                                        ports.Shell.TerminateAllApplications();
                                        Task.Delay(1000).Wait();
                                    }

                                    using (DataflowUtility.LinkBlock("targetLogSource->logMonitoringSource", targetLogSource, logMonitoringSource))
                                    {
                                        var result = ports.Shell.LaunchApplication(program, arguments,
                                            new LaunchApplicationOption()
                                            {
                                                EnableJit = true,
                                                EnableAslr = Arguments.UseAslr,
                                                EnvPath = envDefinitionFile == null ? string.Empty : envDefinitionFile.FullName
                                            });
                                        exitCondition.TargetProcessCompletion = result;

                                        exitStatus = WaitingUtility.WaitExitCondition(logMonitoringSource, exitCondition, ports.CancelTokenSource, coodinator.WaitInterrupt(ports.CancelTokenSource.Token));

                                        Trace.WriteLine("Start to close the target output.");

                                        timeoutForExit.StartExit(TimeSpan.FromSeconds(10));
                                    }

                                    Trace.WriteLine("Unlinked monitoring source.");
                                }

                                Trace.WriteLine("Unlinked target log source.");
                            }
                            finally
                            {
                                targetLogSource.Complete();
                                Trace.WriteLine("Target output closed.");
                            }
                        }
                    }
                }
            }
            finally
            {
                DataflowConsole.Instance.GetConsoleTarget().Completion.Wait(TimeSpan.FromSeconds(5));
                Trace.WriteLine("All outputs closed.");
            }

            timeoutForExit.Finish();

            return exitStatus;
        }

        private ITargetBlock<Tuple<string, string>> GetShellLogTarget(ITargetBlock<Tuple<string, string>> defaultTarget)
        {
            if (MonitorShell)
            {
                return defaultTarget;
            }

            return DataflowBlock.NullTarget<Tuple<string, string>>();
        }
    }
}
