﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Net;
using Nintendo.ControlTarget;
using Nintendo.ControlTarget.ConsoleShell;

namespace Nintendo.RunOnTarget
{
    public class Runner
    {
        public CommonRunCommandArgument Arguments { get; private set; }

        public Runner()
        {
        }

        public ExitStatus Run(RunCommandArgument arguments)
        {
            Arguments = arguments.Common;
            return Run(
                arguments.Common.ProgramOutput,
                arguments.Common.TargetOutput,
                arguments.Common.Program,
                arguments.Common.ExitCondition,
                arguments.TargetName,
                arguments.SuppressAutoKill,
                arguments.ExtraTargetManagerDirectory,
                arguments.Common.Arguments,
                arguments.Common.EnvDefinition,
                arguments.Common.Filter);
        }

        public ExitStatus Run(ITargetBlock<string> output, ITargetBlock<Tuple<string, string>> targetOutput, TargetProgramInfo program, ExitCondition exitCondition, string targetName, bool suppressAutoKill, string extraTargetManagerDirectory, string[] arguments = null, FileInfo envDefinitionFile = null, string filter = null)
        {
            ExitStatus exitStatus;
            var timeoutForExit = new TimeoutForExit();

            try
            {
                using (var targetManager = new TargetManagerAccessor())
                {
                    if (extraTargetManagerDirectory != null)
                    {
                        targetManager.SetExtraTargetManagerDirectory(extraTargetManagerDirectory);
                    }
                    targetManager.EnsureStart();

                    if (Arguments.Reset)
                    {
                        Trace.WriteLine("Reset the target.");
                        using (var target_ = targetManager.GetTarget(targetManager.FindTarget(targetName)))
                        {
                            targetManager.RebootTarget(target_);
                        }

                        Trace.WriteLine("Wait 15 seconds for rebooting the target.");
                        Task.Delay(TimeSpan.FromSeconds(15)).Wait();
                    }

                    if (Arguments.WorkingDirectory != null)
                    {
                        Trace.WriteLine($"Set target working directory. path='{Arguments.WorkingDirectory.FullName}'");
                        using (var target_ = targetManager.GetTarget(targetManager.FindTarget(targetName)))
                        {
                            target_.SetTargetWorkingDirectory(Arguments.WorkingDirectory.FullName);
                        }
                    }

                    var target = targetManager.FindTargetInfo(targetName);
                    var coodinator = new RunnerToolCoordinator(target.GetSerialNumber());

                    using (var targetOwnership = coodinator.RequestOwnership())
                    using (TargetCommunicationInterface ports = new BasicTargetCommunication(targetManager, targetName, filter, Arguments.ConnectTimeout))
                    {
                        var targetLogSource = ports.TargetLog;
                        var logMonitoringSource = new BufferBlock<string>();

                        var targetLogDecorator = DataflowUtility.CreateAdditionMetaBlock("target");

                        try
                        {

                            using (DataflowUtility.LinkBlock("log source -> target log decorator", targetLogSource, targetLogDecorator))
                            using (DataflowUtility.LinkBlock("targetLogSource -> output", targetLogSource, output))
                            using (DataflowUtility.LinkBlock("targetLogDecorator -> targetOutput", targetLogDecorator, targetOutput))
                            {
                                if (!Arguments.Reset && !suppressAutoKill)
                                {
                                    ports.Shell.TerminateAllApplications();
                                    Task.Delay(1000).Wait();
                                }

                                using (DataflowUtility.LinkBlock("targetLogSource->logMonitoringSource", targetLogSource, logMonitoringSource))
                                {
                                    var result = ports.Shell.LaunchApplication(
                                        program,
                                        arguments,
                                        new LaunchApplicationOption() {
                                            EnableJit = true,
                                            EnableAslr = Arguments.UseAslr,
                                            EnvPath = envDefinitionFile == null ? string.Empty : envDefinitionFile.FullName
                                        }
                                    );
                                    exitCondition.TargetProcessCompletion = result;

                                    exitStatus = WaitingUtility.WaitExitCondition(logMonitoringSource, exitCondition, ports.CancelTokenSource, coodinator.WaitInterrupt(ports.CancelTokenSource.Token));

                                    Trace.WriteLine("Start to close the target output.");

                                    timeoutForExit.StartExit(TimeSpan.FromSeconds(10));
                                }

                                Trace.WriteLine("Unlinked monitoring source.");
                            }

                            Trace.WriteLine("Unlinked target log source.");
                        }
                        finally
                        {
                            targetLogSource.Complete();
                            Trace.WriteLine("Target output closed.");
                        }
                    }
                }
            }
            finally
            {
                DataflowConsole.Instance.GetConsoleTarget().Completion.Wait(TimeSpan.FromSeconds(5));
                Trace.WriteLine("All outputs closed.");
            }

            timeoutForExit.Finish();

            return exitStatus;
        }

        private ITargetBlock<Tuple<string, string>> GetShellLogTarget(ITargetBlock<Tuple<string, string>> defaultTarget)
        {
            if (Arguments.Verbose)
            {
                return defaultTarget;
            }

            return DataflowBlock.NullTarget<Tuple<string, string>>();
        }

    }

    public class TimeoutForExit
    {
        public void StartExit(TimeSpan timeout)
        {
            if(EnabledTimeout)
            {
                new Thread(() =>
                {
                    Trace.WriteLine($"Set timer({timeout.TotalSeconds} sec) for exit.");

                    Thread.Sleep(timeout);

                    if (!isFinished)
                    {
                        Console.Error.WriteLine("[ERROR] Failed to exit RunOnTarget.");
                        System.Environment.Exit(1);
                    }
                }).Start();
            }
        }

        public void Finish()
        {
            Trace.WriteLine($"Clear timer for exit.");
            isFinished = true;
        }

        private bool isFinished = false;
        public static bool EnabledTimeout { get; set; } = false;
    }
}
