﻿using System;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace Nintendo.NintendoSdkServiceManager
{
    class TaskTrayApplicationTerminator
    {
        /// <summary>
        /// PID, ファイル名が一致して、かつ最小化状態にあるアプリを終了させます。
        /// </summary>
        public static void Run(int processId, string fileName)
        {
            var targetInfo = new TargetInfo(processId, fileName);

            var infoPtr = Marshal.AllocCoTaskMem(Marshal.SizeOf(targetInfo));
            Marshal.StructureToPtr(targetInfo, infoPtr, false);

            EnumWindows(new EnumWindowsCallback(TerminateTaskTrayApplication), infoPtr);

            Marshal.FreeCoTaskMem(infoPtr);
        }

        private static bool TerminateTaskTrayApplication(IntPtr hWnd, IntPtr lParam)
        {
            var targetInfo = (TargetInfo)Marshal.PtrToStructure(lParam, typeof(TargetInfo));

            if (!IsIconic(hWnd))
            {
                return true;
            }

            int processId;
            GetWindowThreadProcessId(hWnd, out processId);
            if (processId != targetInfo.ProcessId)
            {
                return true;
            }

            var process = Process.GetProcessById(processId);
            if (process.MainModule.FileName != targetInfo.FileName)
            {
                return true;
            }

            SendMessage(hWnd, WM_CLOSE, IntPtr.Zero, IntPtr.Zero);

            return false; // 処理が完了したため、コールバックの呼び出しを中断する。
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct TargetInfo
        {
            public TargetInfo(int processId, string fileName)
            {
                ProcessId = processId;
                FileName = fileName;
            }

            public int ProcessId;
            public string FileName;
        }

        private delegate bool EnumWindowsCallback(IntPtr hWnd, IntPtr lParam);

        const uint WM_CLOSE = 0x10;

        [DllImport("user32.dll")]
        private static extern bool IsIconic(IntPtr hWnd);

        [DllImport("user32.dll")]
        private static extern int SendMessage(IntPtr hWnd, uint Msg, IntPtr wParam, IntPtr lParam);

        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        private extern static bool EnumWindows(EnumWindowsCallback lpEnumFunc, IntPtr lParam);

        [DllImport("user32.dll")]
        private static extern int GetWindowThreadProcessId(IntPtr hWnd, out int lpdwProcessId);
    }
}
