﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public class UsbDeviceAccessor : IDisposable
    {
        private bool isDisposed = false;
        private bool isInitialized = false;
        private const uint DIGCF_PRESENT = 0x00000002;
        private const uint DIGCF_DEVICEINTERFACE = 0x00000010;
        private Guid TargetGuid = new Guid(0x97FFFD48, 0x2D1D, 0x47A0, 0x85, 0xA3, 0x07, 0xFD, 0xE6, 0xFA, 0x01, 0x43);
        private IntPtr Targethandle = INVALID_HANDLE_VALUE;

        static readonly IntPtr INVALID_HANDLE_VALUE = new IntPtr(-1);

        [DllImport("hid.dll")]
        static private extern void HidD_GetHidGuid(ref Guid HidGuid);

        // [DllImport("setupapi.dll")]
        // static private extern IntPtr SetupDiGetClassDevs(
        //     ref Guid ClassGuid, [MarshalAs(UnmanagedType.LPTStr)]string Enumerator, IntPtr hWndParent, uint Flags);
        [DllImport("setupapi.dll")]
        static private extern IntPtr SetupDiGetClassDevs(
            ref Guid ClassGuid, IntPtr Enumerator, IntPtr hWndParent, uint Flags);

        // [DllImport("setupapi.dll", CharSet = CharSet.Auto, SetLastError = true)]
        // static private extern Boolean SetupDiEnumDeviceInterfaces(
        //     IntPtr hDevInfo, ref SP_DEVINFO_DATA devInfo, ref Guid interfaceClassGuid,
        //     UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);
        [DllImport("setupapi.dll", CharSet = CharSet.Auto, SetLastError = true)]
        static private extern Boolean SetupDiEnumDeviceInterfaces(
            IntPtr hDevInfo, IntPtr devInfo, ref Guid interfaceClassGuid,
            UInt32 memberIndex, ref SP_DEVICE_INTERFACE_DATA deviceInterfaceData);

        [DllImport("setupapi.dll", SetLastError = true)]
        public static extern bool SetupDiDestroyDeviceInfoList(
            IntPtr DeviceInfoSet
        );

        [StructLayout(LayoutKind.Sequential)]
        struct SP_DEVICE_INTERFACE_DATA
        {
            internal Int32 cbSize;
            internal Guid InterfaceClassGuid;
            internal Int32 Flags;
            internal UIntPtr Reserved;
        }

        public UsbDeviceAccessor()
        {
            InitializeDevices();
        }

        ~UsbDeviceAccessor()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (isDisposed)
            {
                return;
            }

            if (disposing)
            {
                FinalizeDevices();
            }

            isDisposed = true;
        }

        private bool InitializeDevices()
        {
            if (isInitialized)
            {
                return true;
            }

            Targethandle = SetupDiGetClassDevs(ref TargetGuid, IntPtr.Zero, IntPtr.Zero, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE);
            if (Targethandle == INVALID_HANDLE_VALUE)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "SetupDiGetClassDevs failed");
                return false;
            }

            isInitialized = true;

            return true;
        }

        private bool FinalizeDevices()
        {
            if (!isInitialized)
            {
                return true;
            }

            if (Targethandle != INVALID_HANDLE_VALUE)
            {
                if (!SetupDiDestroyDeviceInfoList(Targethandle))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "SetupDiDestroyDeviceInfoList failed");
                }
            }

            isInitialized = false;
            return true;
        }

        public int GetDeviceNum()
        {
            if (!isInitialized)
            {
                return -1;
            }

            Boolean bResult = true;
            UInt32 deviceIndex = 0;

            while (bResult)
            {
                SP_DEVICE_INTERFACE_DATA interfaceData = new SP_DEVICE_INTERFACE_DATA();
                interfaceData.cbSize = Marshal.SizeOf(interfaceData);

                bResult = SetupDiEnumDeviceInterfaces(Targethandle, IntPtr.Zero, ref TargetGuid, deviceIndex, ref interfaceData);
                if (bResult)
                {
                    deviceIndex++;
                }
            }

            return Convert.ToInt32(deviceIndex);
        }

    }
}
