﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public class SerialNumber
    {
        private string serialNumberWithoutCheckDigit = null;

        private const string serialFormat_SDEV_1_6_2 = @"^XAW([0-9]+)";
        private const string serialFormat_SDEV_1_5   = @"^XAWF([0-9]+)";
        private const string serialFormat_EDEV_MP = @"^XAL([0-9]+)";
        private const string serialFormat_M_SDEV = @"^XK([WJ])([0-9]+)";

        public SerialNumber(string serialString)
        {
            SetSerialString(serialString);
        }

        private SerialNumber()
        {
        }

        ~SerialNumber()
        {
        }

        public string GetString()
        {
            return serialNumberWithoutCheckDigit;
        }

        public string GetStringWithCheckDigit()
        {
            if (serialNumberWithoutCheckDigit == null)
            {
                return null;
            }

            return MakeCheckDigitString(serialNumberWithoutCheckDigit);
        }

        private void SetSerialString(string serialString)
        {
            serialNumberWithoutCheckDigit = ConvertSerialNumberFromString(serialString);
            if (serialNumberWithoutCheckDigit == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Invalid Serial Number String : {0}", serialString);
            }
        }

        private string ConvertSerialNumberFromString(string serialString)
        {
            string serial = serialString.ToUpper();
            Regex regex_SDEV_1_5   = new Regex(serialFormat_SDEV_1_5);
            Regex regex_SDEV_1_6_2 = new Regex(serialFormat_SDEV_1_6_2);
            //TODO: regex_SDEV_1_6_2 との共通化
            Regex regex_EDEV_MP = new Regex(serialFormat_EDEV_MP);
            Regex regex_M_SDEV = new Regex(serialFormat_M_SDEV);

            if (regex_SDEV_1_5.IsMatch(serial))
            {
                Match match = regex_SDEV_1_5.Match(serial);
                string serial_digitString = match.Groups[1].Value;
                if (serial_digitString.Length == 10)
                {
                    return "XAWF" + serial_digitString;
                }
                else if (serial_digitString.Length == 11)
                {
                    return "XAWF" + serial_digitString.Substring(0, 10);
                }
                else
                {
                    return null;
                }
            }
            else if (regex_SDEV_1_6_2.IsMatch(serial))
            {
                Match match = regex_SDEV_1_6_2.Match(serial);
                string serial_digitString = match.Groups[1].Value;
                if (serial_digitString.Length == 10)
                {
                    return "XAW" + serial_digitString;
                }
                else if (serial_digitString.Length == 11)
                {
                    return "XAW" + serial_digitString.Substring(0, 10);
                }
                else
                {
                    return null;
                }
            }
            else if (regex_EDEV_MP.IsMatch(serial))
            {
                Match match = regex_EDEV_MP.Match(serial);
                string serial_digitString = match.Groups[1].Value;
                if (serial_digitString.Length == 10)
                {
                    return "XAL" + serial_digitString;
                }
                else if (serial_digitString.Length == 11)
                {
                    return "XAL" + serial_digitString.Substring(0, 10);
                }
                else
                {
                    return null;
                }
            }
            else if (regex_M_SDEV.IsMatch(serial))
            {
                Match match = regex_M_SDEV.Match(serial);
                string serial_countryString = match.Groups[1].Value;
                string serial_digitString = match.Groups[2].Value;
                if (serial_digitString.Length == 10)
                {
                    return "XK" + serial_countryString + serial_digitString;
                }
                else if (serial_digitString.Length == 11)
                {
                    return "XK" + serial_countryString + serial_digitString.Substring(0, 10);
                }
                else
                {
                    return null;
                }
            }
            return null;
        }

        private string MakeCheckDigitString(string serialWithoutCheckDigit)
        {
            Regex regex_SDEV_1_5 = new Regex(serialFormat_SDEV_1_5);
            Regex regex_SDEV_1_6_2 = new Regex(serialFormat_SDEV_1_6_2);
            Regex regex_EDEV_MP = new Regex(serialFormat_EDEV_MP);
            Regex regex_M_SDEV = new Regex(serialFormat_M_SDEV);
            string serial_digitString = null;

            if (regex_SDEV_1_5.IsMatch(serialWithoutCheckDigit))
            {
                Match match = regex_SDEV_1_5.Match(serialWithoutCheckDigit);
                serial_digitString = match.Groups[1].Value;
            }
            else if (regex_SDEV_1_6_2.IsMatch(serialWithoutCheckDigit))
            {
                Match match = regex_SDEV_1_6_2.Match(serialWithoutCheckDigit);
                serial_digitString = match.Groups[1].Value;
            }
            else if (regex_EDEV_MP.IsMatch(serialWithoutCheckDigit))
            {
                Match match = regex_EDEV_MP.Match(serialWithoutCheckDigit);
                serial_digitString = match.Groups[1].Value;
            }
            else if (regex_M_SDEV.IsMatch(serialWithoutCheckDigit))
            {
                Match match = regex_M_SDEV.Match(serialWithoutCheckDigit);
                serial_digitString = match.Groups[2].Value;
            }

            int digitValue = MakeCheckDigit(serial_digitString);
            if (digitValue >= 0)
            {
                return serialWithoutCheckDigit + digitValue.ToString();
            }
            else
            {
                return null;
            }
        }

        private int MakeCheckDigit(string digitString)
        {
            if (digitString == null || digitString.Length != 10)
            {
                return -1;
            }

            var digits = digitString.Select(e => e - '0').Reverse();
            digits.Select((e, i) => { System.Diagnostics.Debug.WriteLine("sel = {0}, {1}", e, i); return 0; });

            var sum1 = digits.Where((e, i) => (i % 2) == 0).Select(e => { return e; }).Sum();
            var sum2 = digits.Where((e, i) => (i % 2) == 1).Select(e => { return e; }).Sum();
            int digitValue = 10 - (sum1 * 3 + sum2) % 10;
            if (digitValue == 10)
            {
                digitValue = 0;
            }
            return digitValue;
        }

        public bool IsEdev()
        {
            if(serialNumberWithoutCheckDigit == null)
            {
                return false;
            }

            Regex regex_SDEV_1_5 = new Regex(serialFormat_SDEV_1_5);
            Regex regex_SDEV_1_6_2 = new Regex(serialFormat_SDEV_1_6_2);
            Regex regex_EDEV_MP = new Regex(serialFormat_EDEV_MP);
            Regex regex_M_SDEV = new Regex(serialFormat_M_SDEV);

            if (regex_SDEV_1_5.IsMatch(serialNumberWithoutCheckDigit))
            {
                if(serialNumberWithoutCheckDigit.Substring(5, 1) == "3")
                {
                    return true;
                }
            }
            else if (regex_SDEV_1_6_2.IsMatch(serialNumberWithoutCheckDigit))
            {
                if (serialNumberWithoutCheckDigit.Substring(4, 1) == "3")
                {
                    return true;
                }
            }
            else if (regex_EDEV_MP.IsMatch(serialNumberWithoutCheckDigit))
            {
                if (serialNumberWithoutCheckDigit.Substring(4, 1) == "3")
                {
                    return true;
                }
            }
            else if (regex_M_SDEV.IsMatch(serialNumberWithoutCheckDigit))
            {
                if (serialNumberWithoutCheckDigit.Substring(4, 1) == "3")
                {
                    return true;
                }
            }

            return false;
        }

        public override bool Equals(object obj)
        {
            if (obj == null || this.serialNumberWithoutCheckDigit == null)
            {
                return false;
            }

            SerialNumber objSerial = (SerialNumber)obj;
            return this.serialNumberWithoutCheckDigit == objSerial.serialNumberWithoutCheckDigit;
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        public static bool CheckSerialString(string serialString)
        {
            string serial = serialString.ToUpper();
            Regex regex_SDEV_1_5 = new Regex(serialFormat_SDEV_1_5);
            Regex regex_SDEV_1_6_2 = new Regex(serialFormat_SDEV_1_6_2);
            Regex regex_EDEV_MP = new Regex(serialFormat_EDEV_MP);
            Regex regex_M_SDEV = new Regex(serialFormat_M_SDEV);
            if (regex_SDEV_1_5.IsMatch(serial))
            {
                Match match = regex_SDEV_1_5.Match(serial);
                string serial_digitString = match.Groups[1].Value;
                return serial_digitString.Length == 10 || serial_digitString.Length == 11;
            }
            else if (regex_SDEV_1_6_2.IsMatch(serial))
            {
                Match match = regex_SDEV_1_6_2.Match(serial);
                string serial_digitString = match.Groups[1].Value;
                return serial_digitString.Length == 10 || serial_digitString.Length == 11;
            }
            else if (regex_EDEV_MP.IsMatch(serial))
            {
                Match match = regex_EDEV_MP.Match(serial);
                string serial_digitString = match.Groups[1].Value;
                return serial_digitString.Length == 10 || serial_digitString.Length == 11;
            }
            else if (regex_M_SDEV.IsMatch(serial))
            {
                Match match = regex_M_SDEV.Match(serial);
                string serial_digitString = match.Groups[2].Value;
                return serial_digitString.Length == 10 || serial_digitString.Length == 11;
            }
            return false;
        }
    }
}
