﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.Diagnostics;
using Nintendo.Bridge;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public partial class SdevInitializer
    {
        private const int RunOnTargetInterval = 4;
        private const int ResetInterval = 20;

        private string GetVersionString()
        {
            if (Argument.targetFirmwareVersion == null)
            {
                return string.Empty;
            }
            else
            {
                return Argument.targetFirmwareVersion;
            }
        }

        private ExitStatus StopTargetManager()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager.");
            if (ProcessAccessor.StopTargetManager() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private ExitStatus StopTargetManagerForLast()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager .");
            if (ProcessAccessor.StopTargetManagerForLast() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private Dictionary<TargetSpecifier.SdevVersion, string> sdevVersionDictionary = new Dictionary<TargetSpecifier.SdevVersion, string>()
        {
            {TargetSpecifier.SdevVersion.SDEV_1_5, "1.5"},
            {TargetSpecifier.SdevVersion.SDEV_1_6, "1.6"},
            {TargetSpecifier.SdevVersion.SDEV_1_6_G, "1.6+G"},
            {TargetSpecifier.SdevVersion.SDEV_1_8, "1.8"},
            {TargetSpecifier.SdevVersion.SDEV_PRE_MP1, "preMP1"},
            {TargetSpecifier.SdevVersion.SDEV_PRE_MP2, "preMP2"},
            {TargetSpecifier.SdevVersion.SDEV_MP, "MP"},
            {TargetSpecifier.SdevVersion.SDEV_NONE, ""},
            {TargetSpecifier.SdevVersion.SDEV_ERROR, ""},
        };

        private ExitStatus CheckSdevVersion(out TargetSpecifier.SdevVersion sdevVersion, TargetInnerClass targetKey)
        {
            TargetSpecifier.SdevVersion version = TargetSpecifier.GetSdevVersion(targetKey);
            sdevVersion = version;
            switch (version)
            {
                case TargetSpecifier.SdevVersion.SDEV_1_5:
                case TargetSpecifier.SdevVersion.SDEV_1_6:
                case TargetSpecifier.SdevVersion.SDEV_1_6_G:
                case TargetSpecifier.SdevVersion.SDEV_1_8:
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP1:
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP2:
                case TargetSpecifier.SdevVersion.SDEV_MP:
                    LOG.LogLine(LOG_LEVEL.LOG_INFO2, "SDEV version is {0}", sdevVersionDictionary[version]);
                    break;
                case TargetSpecifier.SdevVersion.SDEV_ERROR:
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SDEV version check failed.");
                    return ExitStatus.Failure;
                default:
                    LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Unsupport target.");
                    return ExitStatus.Failure;
            }
            int sleepInterval = 15000;
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Sleep {0} msec.", sleepInterval);
            Thread.Sleep(15000);
            return ExitStatus.Success;
        }

        public ExitStatus ValidateSdevVersion(FirmwareResourceSpecifier firmwareResource)
        {
            // Prod もしくは ProdWithLog の場合はK5鍵の機種のみ利用可能
            if(!firmwareResource.ValidateSetVersion())
            {
                return ExitStatus.Failure;
            }

            return ExitStatus.Success;
        }

        public void RegisterTargetIfTargetTypeIsIp(TargetInnerClass targetKey, string targetManagerPath)
        {
            if(targetKey.Type != TargetInnerClass.TargetType.TargetInner_SdevIp_Force)
            {
                return;
            }

            TargetManagerAccess.RegisterTarget(targetKey.Value, targetManagerPath);
        }

        public ExitStatus UpdateHostBridgeFirmware(TargetInnerClass targetKey, string hostbridgeFirmwarePath, bool recovery = false)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            if (Argument.disableHostBridgeFirmwareUpdate == true)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "skipped HostBridge FW update.");
                return ExitStatus.Success;
            }

            bool specifiedIpAddress = (targetKey.Type == TargetInnerClass.TargetType.TargetInner_SdevIp_Force);
            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "not found target ({0}).", targetKey.Value);
                return ExitStatus.Failure;
            }


            //handle.Release();

            LOG.LogLine("Update HostBridge FW.");

            // Call UpdateHostBridge
            List<string> args1;
            if (!recovery)
            {
                args1 = new List<string>() { "update", "--target", targetIp, "--image", hostbridgeFirmwarePath };
            }
            else
            {
                args1 = new List<string>() { "update", "--target", targetIp, "--image", hostbridgeFirmwarePath, "--recovery" };
            }
            if (specifiedIpAddress == false)
            {
                args1.Add("--waitboot");
            }
            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
#if !CALL_UPDATE_HOSTBRIDGE_EXE
            ProcessAccessor.DumpProcessArgumentList("UpdateHostBridge", argsList);

            int retVar = Nintendo.UpdateHostBridge.CommandInterface.Main(argsList.ToArray());
            if (retVar != 0)
            {
                retVar = Nintendo.UpdateHostBridge.CommandInterface.Main(argsList.ToArray());
            }

            if (specifiedIpAddress == true)
            {
                Thread.Sleep(30000);
            }

            eStatus = retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
#else
            const string updateHostBridgePath = "UpdateHostBridge.exe";
            int exitCode = -1;

            ProcessAccessor.DumpProcessArgumentList("UpdateHostBridge", argsList);

            exitCode = ProcessAccessor.DoProcess(updateHostBridgePath, argsList, OutputDataReceived);
            if (exitCode != 0)
            {
                exitCode = ProcessAccessor.DoProcess(updateHostBridgePath, argsList, OutputDataReceived);
                if (exitCode != 0)
                {
                    goto quit;
                }
            }

            eStatus = ExitStatus.Success;
        quit:
#endif
            if (eStatus == ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "UpdateHostBridge passed.");
            }
            else
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "UpdateHostBridge failed.");
            }
            return eStatus;
        }

        public ExitStatus UpdateRecoveryHostBridgeFirmware(TargetInnerClass targetKey, string hostbridgeRecoveryFirmwarePath)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            // 復旧モード用FWのアップデート処理
            bool required;
            eStatus = IsUpdateHostBridgeRecoveryFirmwareRequired(targetKey, out required);
            if (eStatus != ExitStatus.Success)
            {
                // MEMO: 復旧モード用FWの更新確認に失敗した場合は警告を出力だけ行って初期化は続行させる
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Recovery HostBridge FW check failed, but initialization will continue.");
                return ExitStatus.Success;
            }

            if (required)
            {
                eStatus = UpdateHostBridgeFirmware(targetKey, hostbridgeRecoveryFirmwarePath, true);
                if (eStatus != ExitStatus.Success)
                {
                    // MEMO: 復旧モード用FWの更新に失敗した場合は警告を出力だけ行って初期化は続行させる
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "Recovery HostBridge FW update failed, but initialization will continue.");
                    return ExitStatus.Success;
                }
            }
            return eStatus;
        }

        public ExitStatus IsUpdateHostBridgeRecoveryFirmwareRequired(TargetInnerClass targetKey, out bool required)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "not found target ({0}).", targetKey.Value);
                required = false;
                return ExitStatus.Failure;
            }

            LOG.LogLine("Check HostBridge recovery FW update is required.");

            // Call UpdateHostBridge
            List<string> args1 = new List<string>() { "required", "--target", targetIp };
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("UpdateHostBridge", argsList);

            int retVar = Nintendo.UpdateHostBridge.CommandInterface.Main(argsList.ToArray());
            if(retVar == (int)Nintendo.UpdateHostBridge.CommandInterface.ExitCode.RecoveryUpdateRequired)
            {
                eStatus = ExitStatus.Success;
                required = true;
            }
            else if(retVar == (int)Nintendo.UpdateHostBridge.CommandInterface.ExitCode.RecoveryUpdateNotRequired)
            {
                eStatus = ExitStatus.Success;
                required = false;
            }
            else
            {
                eStatus = ExitStatus.Failure;
                required = false;
            }

            if (eStatus == ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "UpdateHostBridge passed.");
            }
            else
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "UpdateHostBridge failed.");
            }
            return eStatus;
        }

        private ExitStatus UpdateQspiBootImage(TargetInnerClass targetKey, string qspiBootImagePath)
        {
            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                return ExitStatus.Failure;
            }

            LOG.LogLine("Update Recovery boot image.");
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  update target : {0}", targetIp);

            BridgeUpdate update = new BridgeUpdate();

            // InvokeQspiBoot
            bool result = update.UpdateTarget(targetIp, qspiBootImagePath, true);
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  UploadTargetFirmware : {0}", result);
            if (!result)
            {
                return ExitStatus.Failure;
            }

            // Wait
            int sleepInterval = 15000;
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait {0} sec.", sleepInterval/1000);
            Thread.Sleep(sleepInterval);

            return ExitStatus.Success;
        }

#if SYSTEM_UPDATE_IMAGE
        private ExitStatus UpdateSystemImage(string systemUpdaterPath, TargetInnerClass targetKey, string systemImagePath, string targetManagerPath)
#else
        private ExitStatus UpdateSystemImage(string systemUpdaterPath, TargetInnerClass targetKey, string targetManagerPath)
#endif
        {
            LOG.LogLine("Update initial image.");

            string ipAddress = TargetSpecifier.GetTargetIp(targetKey);
            if (ipAddress == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Get IP address failed.");
                return ExitStatus.Failure;
            }

            // call RunOnTarget
            //   MEMO: QSPIブートイメージ書き換え後にはリセットが必要なのでここで行う
            List<string> args1 = new List<string>() {"--target", ipAddress, "--reset",
                                "--failure-timeout", DefaultTimeoutSecond.ToString(),
                                "--pattern-success-exit", "Succeeded initializing the system.", "--extra-targetmanager-dir", targetManagerPath};
            List<string> args2 = new List<string>() { systemUpdaterPath };
#if SYSTEM_UPDATE_IMAGE
            List<string> args3 = new List<string>() { "--", "--input", systemImagePath };
#else
            List<string> args3 = new List<string>();
#endif
            Stopwatch sw = new Stopwatch();

            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
            argsList.AddRange(args2);
            argsList.AddRange(args3);

            ProcessAccessor.DumpProcessArgumentList("RunOnTarget", argsList);

            sw.Start();
            int retVar = Nintendo.RunOnTarget.CommandInterface.Main(argsList.ToArray());
            sw.Stop();

            // MEMO: put dummy line feed for RunOnTarget log
            Console.WriteLine(string.Empty);

            if (retVar != 0)
            {
                // タイムアウトした場合はFuseにより起動できなくなっている可能性がある
                if ((int)sw.Elapsed.TotalSeconds > DefaultTimeoutSecond + RunOnTargetInterval + ResetInterval)    // RunOnTarget の実行時間は timeout より長い
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "If you are about to downgrade firmware, please check downgrade is available.");
                }
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait 10 sec.");
            Thread.Sleep(10000);

            return retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus ExecutePluginsBeforeInitialize()
        {
            bool bRet = Plugins.Execute(Plugins.ExecuteTimingType.ExecutePre, Argument.keeptTargetManagerAlive);
            return bRet ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus ExecutePluginsAfterInitialize(TargetInnerClass targetKey, TargetSpecifier.SdevVersion devkitVersion)
        {
            bool bRet = Plugins.Execute(Plugins.ExecuteTimingType.ExecutePost, Argument.keeptTargetManagerAlive, targetKey, devkitVersion);
            return bRet ? ExitStatus.Success : ExitStatus.Failure;

        }

        private ExitStatus RebootAndRunProgram(string programPath, string ipAddress, string targetKey, string targetManagerPath)
        {
            LOG.LogLine("Run program.");

            // call RunOnTarget
            List<string> args1 = new List<string>() {"--target", ipAddress, "--reset",
                                 "--failure-timeout", DefaultTimeoutSecond.ToString(),
                                 "--pattern-success-exit", "Hello, world.", "--extra-targetmanager-dir", targetManagerPath};
            List<string> args2 = new List<string>() { programPath };

            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
            argsList.AddRange(args2);

            ProcessAccessor.DumpProcessArgumentList("RunOnTarget", argsList);

            int retVar = Nintendo.RunOnTarget.CommandInterface.Main(argsList.ToArray());

            // MEMO: put dummy line feed for RunOnTarget log
            Console.WriteLine(string.Empty);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait 10 sec.");
            Thread.Sleep(10000);

            return retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus RestartTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            // MEMO: This log is disabled because same log is outputted after.
            //LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reboot target.");

            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Get IP address failed.");
                return ExitStatus.Failure;
            }
            TargetManagerAccess.RebootTarget(targetIp, targetManagerPath, Argument.Verbose);

            Thread.Sleep(5000);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reboot target finished.");

            return ExitStatus.Success;
        }

        private ExitStatus ActivateTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            return TargetManagerAccess.ActivateTarget(targetManagerPath, targetKey.Value);
        }

        private ExitStatus WriteSerialNumberToHostBridge(TargetInnerClass targetKey, string targetManagerDirectory, bool writeName)
        {
            switch (targetKey.Type)
            {
                case TargetInnerClass.TargetType.TargetInner_SdevMac:
                    if (TargetSpecifier.GetTargetSerialFromMac(targetKey.Value) != "Unknown")
                    {
                        return ExitStatus.Success;
                    }
                    break;
                case TargetInnerClass.TargetType.TargetInner_SdevIp_Force:
                    if (TargetSpecifier.GetTargetSerialFromIp(targetKey.Value) != "Unknown")
                    {
                        return ExitStatus.Success;
                    }
                    break;
            }

            LOG.LogLine("Write serial number to HostBridge.");

            if (RestartTarget(targetKey, targetManagerDirectory) != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Reboot target failed.");
                return ExitStatus.Failure;
            }

            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Get IP address failed.");
                return ExitStatus.Failure;
            }

            // Call WriteSerialNumber
            List<string> args1 = new List<string>() { "--target", targetIp };
            if (writeName)
            {
                args1.Add("--write-name");
            }
            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("WriteSerialNumber", argsList);

            int retVar = Nintendo.WriteSerialNumber.CommandInterface.Main(argsList.ToArray());

            return retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
        }

        private bool FileCheck(string filePath)
        {
            if (!System.IO.File.Exists(filePath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "file not found : {0}", filePath);
                return false;
            }
            return true;
        }

        private void OutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            Console.WriteLine(e.Data);
        }
    }
}
