﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.IO;

namespace Nintendo.InitializeEdev
{
    public partial class MainForm : Form
    {
        private InitialProgressDialog initialProcessDialog = null;
        private string[] EnumerateNames, EnumerateSerials, EnumerateAddresses;
        private string[] EnumerateSafeModeSerials;
        private List<string> FirmwareVersionList;
        private List<string> FirmwareRevisionList;
        private bool HasOceanFirmware = false;
        private int ConnectedUsbNum = 0;
        private InitializeSdev.ListViewSort LvSorter;

        public MainForm()
        {
            InitializeComponent();
        }

        private void versionToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // MessageBox.Show(Application.ProductName + ' ' + Application.ProductVersion);

            // var assm = System.Reflection.Assembly.GetExecutingAssembly();
            var assm = System.Reflection.Assembly.GetEntryAssembly();
            var name = assm.GetName();
            MessageBox.Show(name.Name + ' ' + name.Version);
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            StartInitializeButton.Enabled = false;
            SaveLogFile.DeleteLogFile();
            UpdateDevkitListFirst();

            this.Enabled = false;  // MEMO: disable for InitialProgressDialog
            BackgroundToolInitializeWorker.RunWorkerAsync();
        }

        private void BackgroundToolInitializeWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            bool abortExit = false;
            initialProcessDialog = new InitialProgressDialog();
            initialProcessDialog.Show(this);

            {
                GetVersionListFirst();

                if (initialProcessDialog.Canceled) goto quit;

                Nintendo.InitializeSdev.ProcessAccessor.StopTargetManager();

                if (initialProcessDialog.Canceled) goto quit;

                if (!UpdateDevkitListCore())
                {
                    abortExit = true;
                    goto quit;
                }
            }

        quit:
            initialProcessDialog.Close();
            if(abortExit)
            {
                Application.Exit();
            }
        }

        private void BackgroundToolInitializeWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {

        }

        private void BackgroundToolInitializeWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            GetVersionListLast();
            UpdateDevkitListLast();
            // MEMO: disable for InitialProgressDialog
            ReScanButton.Enabled = true;
            this.Enabled = true;
            this.TopMost = true;
            this.TopMost = false;
            this.Activate();
        }

        private void ReScanButton_Click(object sender, EventArgs e)
        {
            ReScanButton.Enabled = false;
            UpdateDevkitList();
            ReScanButton.Enabled = true;
        }

        private void StartInitializeButton_Click(object sender, EventArgs e)
        {
            DoInitialize();
        }

        private void DoInitialize()
        {
            if (this.DevkitListView.SelectedItems.Count <= 0)
            {
                MessageBox.Show("EDEV actual equipment is not selected.\nPlease select Devkit.", "Devkit select", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            if (this.firmwareVersionSelectComboBox.SelectedIndex < 0)
            {
                MessageBox.Show("Firmware version field is not selected.\nPlease select .", "EDEV select", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            LogForm f = new LogForm();
            f.EdevSerialNumber = this.DevkitListView.SelectedItems[0].SubItems[2].Text;
            if (this.firmwareVersionSelectComboBox.SelectedIndex == 0)
            {
                f.FirmwareVersion = null;
            }
            else
            {
                f.FirmwareVersion = this.firmwareVersionSelectComboBox.SelectedItem.ToString();
            }
            f.ShowDialog(this);
            f.Dispose();
        }

        private void DevkitListView_SelectedIndexChanged(object sender, EventArgs e)
        {
            StartInitializeButton.Enabled = CanStartInitialize() ? true : false;
            ListViewContextMenuStrip.Enabled = CanStartInitialize() ? true : false;
        }

        private bool CanStartInitialize()
        {
            return DevkitListView.SelectedItems.Count > 0 && firmwareVersionSelectComboBox.SelectedIndex >= 0;
        }

        private void firmwareVersionSelectComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            StartInitializeButton.Enabled = CanStartInitialize() ? true : false;
        }

        private void InitializeDevkitListHeader()
        {
            this.DevkitListView.FullRowSelect = true;
            this.DevkitListView.GridLines = true;

            ColumnHeader columnAddress = new ColumnHeader();
            ColumnHeader columnName = new ColumnHeader();
            ColumnHeader columnSerial = new ColumnHeader();
            // MEMO: width infomation by fontsize of DevkitListView
            //   font  form listview Address Name Serial
            //      9   640      518      80  120    105
            //     10   720      598     90?  140    120
            columnAddress.Text = "Address";
            columnAddress.Width = 80;
            columnName.Text = "Name";
            columnName.Width = 120;
            columnSerial.Text = "Serial";
            columnSerial.Width = 105;
            ColumnHeader[] headerValue = { columnAddress, columnName, columnSerial };
            this.DevkitListView.Columns.AddRange(headerValue);

            LvSorter = new InitializeSdev.ListViewSort();
            LvSorter.ColumnModes = new InitializeSdev.ListViewSort.CompareMode[] {
                                                    InitializeSdev.ListViewSort.CompareMode.Compare_String,
                                                    InitializeSdev.ListViewSort.CompareMode.Compare_String,
                                                    InitializeSdev.ListViewSort.CompareMode.Compare_String };
            this.DevkitListView.ListViewItemSorter = LvSorter;
        }

        private bool UpdateDevkitListFirst()
        {
            this.DevkitListView.Clear();
            InitializeDevkitListHeader();
            this.DevkitListView.Update();

            return true;
        }

        private bool UpdateDevkitListCore()
        {
            EnumerateNames = new string[0];
            EnumerateSerials = new string[0];
            EnumerateAddresses = new string[0];
            EnumerateSafeModeSerials = new string[0];

            Nintendo.InitializeSdev.FirmwareResourceSpecifier firmwareResource = new Nintendo.InitializeSdev.FirmwareResourceSpecifier();
            if (firmwareResource.SetVersion(Nintendo.InitializeSdev.FirmwareResourceSpecifier.GetUpToDateVersion()) == false)
            {
                return false;
            }
            string targetManagerPath = firmwareResource.GetFileNameOfFirm(Nintendo.InitializeSdev.FirmwareType.Host_TargetManager);
            Nintendo.InitializeSdev.TargetManagerAccess.EnumerateUsbTargets(targetManagerPath, out EnumerateNames, out EnumerateSerials, out EnumerateAddresses);

            string recoveryWriterUsbPath = firmwareResource.GetFileNameOfFirm(Nintendo.InitializeSdev.FirmwareType.Host_RecoveryWriterUsb);
            if(!File.Exists(recoveryWriterUsbPath))
            {
                ShowRecoveryWriterUsbError(recoveryWriterUsbPath);
                return false;
            }
            if (SafeMode.GetSafeModeSerialNumbers(recoveryWriterUsbPath) == ExitStatus.Success)
            {
                SafeMode.EnumerateSafeModeSerialNumbers(out EnumerateSafeModeSerials);
            }

            return true;
        }

        private bool UpdateDevkitListLast()
        {
#if false
            if (ConnectedUsbNum > 1)
            {
                // USB multiple connect check
                MessageBox.Show(this, string.Format("{0} NX devkits are connected via USB.\nPlease disconnect other ones.", ConnectedUsbNum),
                                    "USB connect error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return true;
            }
            else if (ConnectedUsbNum < 0)
#else
            if (ConnectedUsbNum < 0)
#endif
            {
                return true;
            }

            if (EnumerateSerials == null)
            {
                return true;
            }

            for (int i = 0; i < EnumerateSerials.Length; i++)
            {
                // System.Diagnostics.Debug.WriteLine("target : {0}, {1}, {2}", EnumerateNames[i], EnumerateSerials[i], EnumerateAddresses[i]);
                string[] sItems = { EnumerateAddresses[i], EnumerateNames[i], EnumerateSerials[i] };
                ListViewItem lItem = new ListViewItem(sItems);
                lItem.Tag = i;
                this.DevkitListView.Items.Add(lItem);
            }
            for (int i = 0; i < EnumerateSafeModeSerials.Length; i++)
            {
                string[] sItems = { "USB", "(Safe Mode)", EnumerateSafeModeSerials[i] };
                ListViewItem lItem = new ListViewItem(sItems);
                lItem.Tag = i;
                this.DevkitListView.Items.Add(lItem);
            }
            // for debug of DevkitListView's column width
            // string[] testitem = { "USB", "AA-AA-AA-AA-AA-AA", "XAWF0111100000X" };
            // this.DevkitListView.Items.Add(new ListViewItem(testitem));

            return true;
        }

        private bool UpdateDevkitList()
        {
            if (!UpdateDevkitListFirst())
            {
                return false;
            }

            if (!UpdateDevkitListCore())
            {
                return false;
            }

            if (!UpdateDevkitListLast())
            {
                return false;
            }

            return true;
        }

        private void GetVersionListFirst()
        {
            FirmwareVersionList = new List<string>();
            FirmwareRevisionList = new List<string>();
            bool bRet = Nintendo.InitializeSdev.FirmwareResourceSpecifier.EnumerateAllVersions(ref FirmwareVersionList);
            Nintendo.InitializeSdev.FirmwareResourceSpecifier.EnumerateAllRevisions(ref FirmwareRevisionList);
            HasOceanFirmware = InitializeSdev.FirmwareResourceSpecifier.ContainsOceanFirmware();
        }

        private void GetVersionListLast()
        {
            this.firmwareVersionSelectComboBox.Items.Clear();
            int index = 0;
            foreach (string versionStr in FirmwareVersionList)
            {
                string revisionStr = FirmwareRevisionList[index];
                if (revisionStr != string.Empty)
                {
                    this.firmwareVersionSelectComboBox.Items.Add(string.Format("{0}  (rev:{1})", versionStr, revisionStr));
                }
                else
                {
                    this.firmwareVersionSelectComboBox.Items.Add(versionStr);
                }

                if (HasOceanFirmware)
                {
                    if (InitializeSdev.FirmwareResourceAccessor.GetInstance().CheckOceanVersion(versionStr))
                    {
                        firmwareVersionSelectComboBox.SelectedIndex = index;
                    }
                }
                else
                {
                    firmwareVersionSelectComboBox.SelectedIndex = 0;
                }
                index++;
            }
        }

        private void GetVersionList()
        {
            GetVersionListFirst();
            GetVersionListLast();
        }

        private void viewhelpToolStripMenuItem_Click(object sender, EventArgs e)
        {
            string itemPath = ConstData.GetInitializeEdevDocumentUrl();
            if(File.Exists(itemPath))
            {
                System.Diagnostics.Process.Start(itemPath);
            }
            else
            {
                MessageBox.Show("Document page not found.\nPlease install Document Package.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        private void viewEDEVtroubleShootingToolStripMenuItem_Click(object sender, EventArgs e)
        {
            string itemPath = ConstData.GetErrorDocumentUrl();
            if (File.Exists(itemPath))
            {
                System.Diagnostics.Process.Start(itemPath);
            }
            else
            {
                MessageBox.Show("Document page not found.\nPlease install Document Package.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        private void DevkitListView_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            LvSorter.Column = e.Column;
            this.DevkitListView.Sort();
        }

        private void startinitializationToolStripMenuItem_Click(object sender, EventArgs e)
        {
            DoInitialize();
        }

        private void ShowRecoveryWriterUsbError(string filePath)
        {
            string errorMsg = string.Format("File is not found. ({0})", filePath);
            MessageBox.Show(errorMsg, "error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }
}
