﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.ControlTarget;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    public class TargetManagerAccess
    {
        public static ExitStatus UnregisterTarget(string targetManagerDirectory, string targetArgumentString, EdevInitializeCommandArgument.TargetArgumentType taragetArgumentType, bool isVerbose)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Unregister target ('{0}', {1}) in NintendoTargetManager.", targetArgumentString, taragetArgumentType.ToString());

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();

                // MEMO: TM起動直後では登録に失敗することがありましたので調整しました (2017.07.25)
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Wait 5 sec for preparing to unregister.");
                System.Threading.Thread.Sleep(5000);

                if (taragetArgumentType == EdevInitializeCommandArgument.TargetArgumentType.TypeOther)
                {
                    return ExitStatus.Failure;
                }
                else
                {
                    string searchString = targetArgumentString;
                    if(targetArgumentString.ToLower() != "usb")
                    {
                        if (InitializeSdev.SerialNumber.CheckSerialString(targetArgumentString))
                        {
                            InitializeSdev.SerialNumber serial = new InitializeSdev.SerialNumber(targetArgumentString);
                            searchString = serial.GetStringWithCheckDigit();
                        }
                    }
                    Nintendo.ControlTarget.Tmapi.RegisteredTargetInfo registerdInfo = tmapiAccessor.FindRegisteredTarget(searchString);
                    if (registerdInfo != null)
                    {
                        tmapiAccessor.Unregister(registerdInfo.GetTargetHandle());
                    }
                }

                return ExitStatus.Success;
            }
        }

        public static ExitStatus RegisterTarget(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey) // , bool isVerbose)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Register target ('{0}') in NintendoTargetManager.", targetKey.Value);

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();

#if false
                // MEMO: TM起動直後では登録に失敗することがありましたので調整しました (2017.07.25)
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Wait 5 sec for preparing to unregister.");
                System.Threading.Thread.Sleep(5000);
#endif

                if (targetKey.Type != InitializeSdev.TargetInnerClass.TargetType.TargetInner_EdevSerial)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Not support target key type : {0}", targetKey.Type.ToString());
                    return ExitStatus.Failure;
                }

                string searchString = targetKey.Value;
                if (!InitializeSdev.SerialNumber.CheckSerialString(searchString))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Not serial number string : {0}", searchString);
                    return ExitStatus.Failure;
                }

                InitializeSdev.SerialNumber serial = new InitializeSdev.SerialNumber(searchString);
                searchString = serial.GetStringWithCheckDigit();
                Nintendo.ControlTarget.Tmapi.TargetInfo targetInfo = tmapiAccessor.DetectTarget(searchString);
                if (targetInfo != null)
                {
                    tmapiAccessor.Register(targetInfo);
                }

                return ExitStatus.Success;
            }
        }

        public static ExitStatus ActivateTarget(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Set target active in NintendoTargetManager ('{0}') .", targetKey.Value);

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                var registerdInfo = GetTargetInfo(targetManagerDirectory, targetKey, tmapiAccessor);

                if (registerdInfo == null)
                {
                    return ExitStatus.Failure;
                }
                var targetAccessor = tmapiAccessor.GetTarget(registerdInfo.GetTargetHandle());
                targetAccessor.SetDefault();

                return ExitStatus.Success;
            }
        }

        public static ExitStatus GetDevkitHardwareType(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey, out string devkitHardwareType)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Get DevKit hardware type from NintendoTargetManager ('{0}') .", targetKey.Value);

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                // ターゲットの存在確認
                // (ちゃんとUSB認識されていることを確認する)
                ControlTarget.Tmapi.TargetInfo checkTargetInfo = DetectTargetInfo(targetManagerDirectory, targetKey, tmapiAccessor);
                if(checkTargetInfo == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Target cannnot be detected. ({0})", targetKey.Value);
                    devkitHardwareType = null;
                    return ExitStatus.Failure;
                }

                // 登録されているターゲットを取得しハードウェアタイプを取得する
                var registerdInfo = GetTargetInfo(targetManagerDirectory, targetKey, tmapiAccessor);

                if (registerdInfo == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Not found target '{0}' .", targetKey.Value);
                    devkitHardwareType = null;
                    return ExitStatus.Failure;
                }
                var targetInfo = tmapiAccessor.GetTargetInfo(registerdInfo.GetTargetHandle());
                devkitHardwareType = targetInfo.GetHardwareType();

                return ExitStatus.Success;
            }
        }

        public static ExitStatus ConnectTarget(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Connect target in NintendoTargetManager ('{0}') .", targetKey.Value);

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                // var targetAccessor = tmapiAccessor.ConnectTarget(targetKey.Value);
                var targetInfo = DetectTargetInfo(targetManagerDirectory, targetKey, tmapiAccessor);
                if(targetInfo == null)
                {
                    return ExitStatus.Failure;
                }
                // var targetAccessor = tmapiAccessor.GetTarget(targetInfo.)
                return ExitStatus.Success;
            }
        }

        public static ExitStatus DisconnectTarget(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Disconnect target in NintendoTargetManager ('{0}') .", targetKey.Value);

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                var registerdInfo = GetTargetInfo(targetManagerDirectory, targetKey, tmapiAccessor);

                if (registerdInfo == null)
                {
                    return ExitStatus.Failure;
                }
                var targetAccessor = tmapiAccessor.GetTarget(registerdInfo.GetTargetHandle());
                targetAccessor.Disconnect();

                return ExitStatus.Success;
            }
        }

        public static ControlTarget.Tmapi.RegisteredTargetInfo GetTargetInfo(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey, TargetManagerAccessor tmapiAccessor)
        {
            tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
            tmapiAccessor.SetStartTrayIcon(true);
            tmapiAccessor.EnsureStart();

            ControlTarget.Tmapi.RegisteredTargetInfo registerdInfo = null;
            if (targetKey.Type == InitializeSdev.TargetInnerClass.TargetType.TargetInner_EdevSerial)
            {
                InitializeSdev.SerialNumber serial = new InitializeSdev.SerialNumber(targetKey.Value);
                if (serial.GetString() != null)
                {
                    registerdInfo = tmapiAccessor.FindRegisteredTarget(serial.GetStringWithCheckDigit());
                }
            }
            else if (targetKey.Type == InitializeSdev.TargetInnerClass.TargetType.TargetInner_EdevConnectedUsb)
            {
                registerdInfo = tmapiAccessor.FindRegisteredFirstUsbTargets();
            }

            return registerdInfo;
        }

        private static ControlTarget.Tmapi.TargetInfo DetectTargetInfo(string targetManagerDirectory, InitializeSdev.TargetInnerClass targetKey, TargetManagerAccessor tmapiAccessor)
        {
            tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
            tmapiAccessor.SetStartTrayIcon(true);
            tmapiAccessor.EnsureStart();

            ControlTarget.Tmapi.TargetInfo targetInfo = null;
            if (targetKey.Type == InitializeSdev.TargetInnerClass.TargetType.TargetInner_EdevSerial)
            {
                InitializeSdev.SerialNumber serial = new InitializeSdev.SerialNumber(targetKey.Value);
                if (serial.GetString() != null)
                {
                    targetInfo = tmapiAccessor.DetectTarget(serial.GetStringWithCheckDigit());
                }
            }

            return targetInfo;
        }
    }
}
