﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Xml;
using Microsoft.Build.Construction;
using Microsoft.Build.BuildEngine;
using Microsoft.Build.Evaluation;
using Microsoft.Build.Framework;
using Project = Microsoft.Build.Evaluation.Project;

using LOG = Nintendo.CopyCliCoreLibrary.Logger;
using LOG_LEVEL = Nintendo.CopyCliCoreLibrary.Logger.Level;

namespace Nintendo.CopyCliCoreLibrary
{
    class ParseProject
    {
        List<ProjectInfo> projectInfoList = new List<ProjectInfo>();

        struct ProjectInfo
        {
            public string m_Name;
            public string m_ProjectFilePath;
            public string m_TargetDirectoryPath;
            public List<DependetProject> m_ReferenceProjectPathList;
            public bool m_IsCliDependentChecked;
            public bool m_IsCliDependent;
        }

        struct DependetProject
        {
            public string m_Name;
            // public string m_Guid;    // MEMO: vcxprojの依存解析が必要ないならば不要
        }

        Dictionary<string, string> BuildPlatformDictionary = new Dictionary<string,string>
                                                                {
                                                                    {"x86", "Win32"},
                                                                    {"x64", "x64"},
                                                                    {"Any CPU", "Any CPU"},
                                                                };

        public bool CopyDependentDllOfCliLibrary(string mainSlnFilePath, string substituteSlnFilePath, string cliProjectFilePath, string dependProjectFilePath, string buildConfiguration, string buildPlatform)
        {
            // MEMO: MsBuild APIを呼び出した時に .sln.cache ファイルを生成しないようにする
            Environment.SetEnvironmentVariable("MSBuildUseNoSolutionCache", "1", EnvironmentVariableTarget.Process);

            if(!FileCheck(cliProjectFilePath) || !FileCheck(dependProjectFilePath))
            {
                return false;
            }

            string slnFilePath = SelectSolutionFile(mainSlnFilePath, substituteSlnFilePath);
            if(slnFilePath == null)
            {
                return false;
            }

            if (!ParseSolution(slnFilePath, buildConfiguration, buildPlatform))
            {
                return false;
            }

            if (!CheckCliDependet(cliProjectFilePath, dependProjectFilePath))
            {
                return false;
            }

            if (!CopyDependentOutput(dependProjectFilePath, buildConfiguration, buildPlatform))
            {
                return false;
            }

            return true;
        }

        private string SelectSolutionFile(string mainSlnFilePath, string substituteSlnFilePath)
        {
            if (FileCheck(mainSlnFilePath, false) && Path.GetExtension(mainSlnFilePath).ToLower() == ".sln")
            {
                return mainSlnFilePath;
            }
            else
            {
                LOG.LogLine("The main sln file is not found, so used substitute path.");
                if (FileCheck(substituteSlnFilePath))
                {
                    return substituteSlnFilePath;
                }
            }
            return null;
        }

        private bool ParseSolution(string slnFilePath, string buildConfiguration, string buildPlatform)
        {
            // MEMO: この実装方法では .sln ファイルの中の .csproj のみ検索できて .vcxproj は検索できない
            //       .vcsproj も検索するようにする場合はこの関数をそのように書き換えた後にビルドオプション USE_VCXPROJ_SOLUTION_PARSE を有効にする必要がある
            //        (でもそれはしなくとも十分仕様は満たされる)
            string wrapperContent = SolutionWrapperProject.Generate(slnFilePath, toolsVersionOverride: null, projectBuildEventContext: null);
            byte[] rawWrapperContent = Encoding.Unicode.GetBytes(wrapperContent.ToCharArray());

            IDictionary<string, string> globalProperties = new Dictionary<string, string>();
            globalProperties.Add("Configuration", buildConfiguration);
            globalProperties.Add("Platform", buildPlatform);

            using (MemoryStream memStream = new MemoryStream(rawWrapperContent))
            using (XmlTextReader xmlReader = new XmlTextReader(memStream))
            {
                Project proj = ProjectCollection.GlobalProjectCollection.LoadProject(xmlReader, globalProperties, null);
                foreach (ProjectItem referencedProjectNode in GetOrderedProjectReferencesFromWrapper(proj))
                {
                    string referencedProjectPath = Path.Combine(Path.GetDirectoryName(slnFilePath), referencedProjectNode.EvaluatedInclude);
                    Project referencedProject = ProjectCollection.GlobalProjectCollection.LoadProject(referencedProjectPath, globalProperties, null);

                    ProjectInfo pInfo = new ProjectInfo();
                    pInfo.m_Name = GetProjectName(referencedProject);
                    pInfo.m_ProjectFilePath = referencedProjectPath;
                    pInfo.m_TargetDirectoryPath = GetTargetDirectory(referencedProject);
                    pInfo.m_ReferenceProjectPathList = GetDependentList(referencedProject);
                    // System.Diagnostics.Debug.WriteLine(string.Format("{0}", pInfo.m_Name));
                    projectInfoList.Add(pInfo);
                }
                ProjectCollection.GlobalProjectCollection.UnloadAllProjects();
            }

            return true;
        }

        private IEnumerable<ProjectItem> GetOrderedProjectReferencesFromWrapper(Project solutionWrapperProject)
        {
            int buildLevel = 0;
            while (true)
            {
                var items = solutionWrapperProject.GetItems(String.Format("BuildLevel{0}", buildLevel.ToString()));
                if (items.Count == 0)
                {
                    yield break;
                }
                foreach (var item in items)
                {
                    yield return item;
                }
                buildLevel++;
            }
        }

        private string GetTargetDirectory(Project project)
        {
            return project.GetProperty("TargetDir").EvaluatedValue;
        }

        private string GetProjectName(Project project)
        {
            return project.GetPropertyValue("ProjectName");
        }

        private List<DependetProject> GetDependentList(Project project)
        {
            List<DependetProject> depList = new List<DependetProject>();

            ICollection<ProjectItem> itemList = project.Items;
            foreach (ProjectItem item in itemList)
            {
                if (item.HasMetadata("ReferenceSourceTarget"))
                {
                    ICollection<ProjectMetadata> metaDataCollection = item.Metadata;
                    foreach (ProjectMetadata pMeta in metaDataCollection)
                    {
                        if (pMeta.Name == "Name")
                        {
                            string dependentName = pMeta.EvaluatedValue;
                            DependetProject dependentProject = new DependetProject();
                            dependentProject.m_Name = dependentName;
                            depList.Add(dependentProject);
                        }
                    }
                }
            }

            return depList;
        }

        // 現状 USE_VCXPROJ_SOLUTION_PARSE を利用することはない
#if !USE_VCXPROJ_SOLUTION_PARSE
        private bool CheckCliDependet(string cliProjectFilePath, string dependProjectFilePath)
        {
            string cliProjectName = GetNameFromProject(cliProjectFilePath);

            for (int i=0; i< projectInfoList.Count; i++)
            {
                ProjectInfo projectInfo = projectInfoList[i];

                if (!CheckCliDependetRecursive(ref projectInfo, cliProjectName))
                {
                    return false;
                }
                projectInfoList[i] = projectInfo;
            }

            return true;
        }

        private bool CheckCliDependetRecursive(ref ProjectInfo pInfo, string cliProjectName)
        {
            if (pInfo.m_IsCliDependentChecked)
            {
                return true;
            }

            // 依存リストを全てチェックして依存するものがあればこのプロジェクトも依存
            // MEMO: リストの循環チェックは行わない。Visual Studio がビルド時にチェックしてくれるため
            foreach(DependetProject dependProjct in pInfo.m_ReferenceProjectPathList)
            {
                if(dependProjct.m_Name == cliProjectName)
                {
                    pInfo.m_IsCliDependent = true;
                    pInfo.m_IsCliDependentChecked = true;
                    return true;
                }

                ProjectInfo dependProjInfo = new ProjectInfo();
                if(GetProjectInfoFromPath(out dependProjInfo, dependProjct.m_Name))
                {
                    if(dependProjInfo.m_IsCliDependentChecked)
                    {
                        if (dependProjInfo.m_IsCliDependent)
                        {
                            pInfo.m_IsCliDependent = true;
                            pInfo.m_IsCliDependentChecked = true;
                            return true;
                        }
                    }
                    else
                    {
                        if(CheckCliDependetRecursive(ref dependProjInfo, cliProjectName))
                        {
                            if (dependProjInfo.m_IsCliDependent)
                            {
                                pInfo.m_IsCliDependent = true;
                                pInfo.m_IsCliDependentChecked = true;
                                return true;
                            }
                        }
                        else
                        {
                            return false;
                        }
                    }
                }
                else
                {
                    // vcxproj の場合は見つからないのでエラーではない
                }
            }

            // そうでなければ依存していない
            pInfo.m_IsCliDependent = false;
            pInfo.m_IsCliDependentChecked = true;
            return true;
        }

        private string GetNameFromProject(string projectFilePath)
        {
            Project referencedProject = ProjectCollection.GlobalProjectCollection.LoadProject(projectFilePath);
            string projectName = GetProjectName(referencedProject);

            ProjectCollection.GlobalProjectCollection.UnloadAllProjects();

            return projectName;
        }
#else
        private bool CheckCliDependet(string cliProjectFilePath, string dependProjectFilePath)
        {
            for (int i = 0; i < projectInfoList.Count; i++)
            {
                ProjectInfo projectInfo = projectInfoList[i];

                if (!CheckCliDependetRecursive(ref projectInfo, cliProjectFilePath, dependProjectFilePath))
                {
                    return false;
                }
                projectInfoList[i] = projectInfo;
            }

            return true;
        }

        private bool CheckCliDependetRecursive(ref ProjectInfo pInfo, string cliProjectFilePath, string dependProjectFilePath)
        {
            if (pInfo.m_IsCliDependentChecked)
            {
                return true;
            }

            if(pInfo.m_ProjectFilePath == cliProjectFilePath)
            {
                pInfo.m_IsCliDependent = true;
                pInfo.m_IsCliDependentChecked = true;
                return true;
            }
            else if(pInfo.m_ProjectFilePath == dependProjectFilePath)
            {
                pInfo.m_IsCliDependent =false;
                pInfo.m_IsCliDependentChecked = true;
                return true;
            }

            // 依存リストを全てチェックして依存するものがあればこのプロジェクトも依存
            // MEMO: リストの循環チェックは行わない。Visual Studio がビルド時にチェックしてくれるため
            foreach(DependetProject dependProjct in pInfo.m_ReferenceProjectPathList)
            {
                ProjectInfo dependProjInfo = new ProjectInfo();
                if(GetProjectInfoFromPath(out dependProjInfo, dependProjct.m_Name))
                {
                    if(dependProjInfo.m_IsCliDependentChecked)
                    {
                        if (dependProjInfo.m_IsCliDependent)
                        {
                            pInfo.m_IsCliDependent = true;
                            pInfo.m_IsCliDependentChecked = true;
                            return true;
                        }
                    }
                    else
                    {
                        if(CheckCliDependetRecursive(ref dependProjInfo, cliProjectFilePath, dependProjectFilePath))
                        {
                            if (dependProjInfo.m_IsCliDependent)
                            {
                                pInfo.m_IsCliDependent = true;
                                pInfo.m_IsCliDependentChecked = true;
                                return true;
                            }
                        }
                        else
                        {
                            return false;
                        }
                    }
                }
                else
                {
                    return false;
                }
            }

            // そうでなければ依存していない
            pInfo.m_IsCliDependent = false;
            pInfo.m_IsCliDependentChecked = true;
            return true;
        }
#endif

        private bool GetProjectInfoFromPath(out ProjectInfo projectInfo, string projectName)
        {
            foreach(ProjectInfo pInfo in projectInfoList)
            {
                if(pInfo.m_Name == projectName)
                {
                    projectInfo = pInfo;
                    return true;
                }
            }
            projectInfo = new ProjectInfo();
            return false;
        }

        private bool CopyDependentOutput(string dependProjectFilePath, string buildConfiguration, string buildPlatform)
        {
            string copySrcPath, copyPdbSrcPath;
            if(!GetOutputPathFromProject(out copySrcPath, dependProjectFilePath, buildConfiguration, buildPlatform))
            {
                return false;
            }

            GetPdbPath(out copyPdbSrcPath, copySrcPath);

            foreach(ProjectInfo pInfo in projectInfoList)
            {
                if(pInfo.m_IsCliDependent)
                {
                    string copyDestPath = pInfo.m_TargetDirectoryPath;
                    // LOG.LogLine("Copy file: {0} to {1}", copySrcPath, copyDestPath);
                    if(!FileCopyToDirectory(copyDestPath, copySrcPath))
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        private bool GetOutputPathFromProject(out string targetPath, string projectFilePath, string buildConfiguration, string buildPlatform)
        {
            IDictionary<string, string> globalProperties = new Dictionary<string, string>();
            globalProperties.Add("Configuration", buildConfiguration);
            globalProperties.Add("Platform", BuildPlatformDictionary[buildPlatform]);

            Project referencedProject = ProjectCollection.GlobalProjectCollection.LoadProject(projectFilePath, globalProperties, null);

            targetPath = referencedProject.GetPropertyValue("TargetPath");

            ProjectCollection.GlobalProjectCollection.UnloadAllProjects();

            return true;
        }

        private bool GetPdbPath(out string pdbPath, string dllPath)
        {
            string filePath = Path.ChangeExtension(dllPath, "pdb");
            if(!File.Exists(filePath))
            {
                pdbPath = null;
                return false;
            }
            pdbPath = filePath;
            return true;
        }

        private bool FileCopyToDirectory(string destDirectoryPath, string srcFilePath)
        {
            // .dll ファイルをコピーする
            string fileName = Path.GetFileName(srcFilePath);
            string destFilePath = Path.Combine(destDirectoryPath, fileName);

            // LOG.LogLine(LOG_LEVEL.LOG_INFO, "copy {0} to {1}", srcFilePath, destFilePath);
            if(!FileCheck(srcFilePath))
            {
                return false;
            }

            if(!Directory.Exists(destDirectoryPath))
            {
                Directory.CreateDirectory(destDirectoryPath);
            }

            File.Copy(srcFilePath, destFilePath, true);

            // .pdb ファイルも(あれば)コピーする
            string pdbFilePath;
            if(GetPdbPath(out pdbFilePath, srcFilePath))
            {
                fileName = Path.GetFileName(pdbFilePath);
                string pdbDestFilePath = Path.Combine(destDirectoryPath, fileName);
                File.Copy(pdbFilePath, pdbDestFilePath, true);
            }

            return true;
        }

        private bool FileCheck(string filePath, bool errorLog = true)
        {
            if (!File.Exists(filePath) && errorLog)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "File not found : {0}", filePath);
                return false;
            }
            return true;
        }
    }
}
