﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Threading;
using System.Xml;
using System.Xml.Linq;

namespace Nintendo.ControlTarget
{
    public class Utility
    {
        public class TargetInformationFromTargetManager
        {
            public string name = "";
            public string ipAddress = "";
            public string tmaVersion = "";
            public string macAddress = "";
            public string spec = "";
            public string connectionType = "";
            public string hardwareType = "";
            public string serialNumber = "";
        }

        public const String HtcsSystemPortPrefix = @"iywys@$";

        public const String controlTargetDetectMatcher =
        @"(?<name>[A-Za-z0-9_-]+|<NoName>)\t" +
        @"(?<ipAddress>[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+|<NoIpAddress>)\t" +
        @"(?<tmaVersion>[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+|<NoTmaVersion>)\t" +
        @"(?<macAddress>[0-9A-Fa-f]{2}-[0-9A-Fa-f]{2}-[0-9A-Fa-f]{2}-[0-9A-Fa-f]{2}-[0-9A-Fa-f]{2}-[0-9A-Fa-f]{2}|<NoMacAddress>)\t" +
        @"(?<connectionType>\w+|<NoConnectionType>)\t" +
        @"(?<hardwareType>(\w|_)+|<NoHardwareType>)\t" +
        @"(?<serialNumber>[A-Za-z0-9]+|<NoSerialNumber>|Corrupted S/N)(\r\n|$)";

        public static string MakeJoinedPattern(string[] patterns)
        {
            return string.Join("|", (from pattern in patterns select "(" + pattern + ")").ToList());
        }

        public static string FormatXml(XmlDocument xmlDocument)
        {
            var settings = new XmlWriterSettings()
            {
                Encoding = Encoding.Default,
                Indent = true,
                IndentChars = "    "
            };

            var stream = new MemoryStream();
            var writer = XmlWriter.Create(stream, settings);
            var reader = new System.IO.StreamReader(stream);

            xmlDocument.Save(writer);

            stream.Position = 0;

            return reader.ReadToEnd();
        }

        public static string FormatXml(XDocument xDocument)
        {
            var settings = new XmlWriterSettings()
            {
                Encoding = Encoding.UTF8,
                Indent = true,
                IndentChars = "  "
            };

            using (var stream = new MemoryStream())
            {
                using (var writer = XmlWriter.Create(stream, settings))
                {
                    xDocument.Save(writer);
                }

                stream.Position = 0;

                using (var reader = new System.IO.StreamReader(stream))
                {
                    return reader.ReadToEnd();
                }
            }
        }

        public static string FormatXml(string xmlText)
        {
            var xml = new XmlDocument();
            xml.LoadXml(xmlText);

            return FormatXml(xml);
        }

        public static string ConvertToCsv(IEnumerable<IEnumerable<string>> table)
        {
            var sb = new StringBuilder();

            foreach (var line in table)
            {
                sb.AppendLine(string.Join(",", (from element in line select ConvertToCsvElement(element) as string).ToArray()));
            }

            return sb.ToString();
        }

        public static string ConvertToCsvElement(string element)
        {
            return string.Format("\"{0}\"", element.Replace("\"", "\"\""));
        }

        public static int FindPortForService(string serviceName, string targetID, bool enableSystemPort)
        {
            using (var htcDaemonAccessor = new TargetManagerAccessor())
            {
                htcDaemonAccessor.EnsureStart();

                var services = htcDaemonAccessor.ListService();

                foreach (var service in services)
                {
                    if (!enableSystemPort && service.ServiceName.StartsWith(Utility.HtcsSystemPortPrefix))
                    {
                        continue;
                    }

                    // There may be multiple targets. Only match if we're looking at the correct target.
                    if (service.ServiceName.Equals(serviceName) && service.TargetName.Equals(targetID))
                    {
                        return service.EndPoint.Port;
                    }
                }

                return -1;
            }
        }

        public static void PrintTargetInformation(TargetInformationFromTargetManager info)
        {
            Console.Out.Write(info.name + ", ");
            Console.Out.Write(info.ipAddress + ", ");
            Console.Out.Write(info.tmaVersion + ", ");
            Console.Out.Write(info.macAddress + ", ");
            Console.Out.Write(info.connectionType + ", ");
            Console.Out.Write(info.hardwareType + ", ");
            Console.Out.Write(info.serialNumber);
            Console.Out.WriteLine();
        }

        public static TargetInformationFromTargetManager[] GetTargetInformations()
        {
            var timeout = TimeSpan.FromSeconds(5);
            var targets = "";

            int retryCount = 3;
            for (int i = 0; i < retryCount; i++)
            {
                targets = ListTargetCommand.Run(true, timeout, true);
                if (!String.IsNullOrEmpty(targets))
                {
                    break;
                }
            }
            if (String.IsNullOrEmpty(targets))
            {
                return null;
            }

            var matched = Regex.Matches(targets, controlTargetDetectMatcher);
            TargetInformationFromTargetManager[] targetInfo = new TargetInformationFromTargetManager[matched.Count];
            var index = 0;
            foreach (Match i in matched)
            {
                targetInfo[index] = new TargetInformationFromTargetManager();
                targetInfo[index].name = i.Groups["name"].Value;
                targetInfo[index].ipAddress = i.Groups["ipAddress"].Value;
                targetInfo[index].tmaVersion = i.Groups["tmaVersion"].Value;
                targetInfo[index].macAddress = i.Groups["macAddress"].Value;
                targetInfo[index].spec = i.Groups["spec"].Value;
                targetInfo[index].connectionType = i.Groups["connectionType"].Value;
                targetInfo[index].hardwareType = i.Groups["hardwareType"].Value;
                targetInfo[index].serialNumber = i.Groups["serialNumber"].Value;
                index++;
            }

            return targetInfo;
        }

        public static bool IsIPAddress(string target)
        {
            IPAddress targetAsIpAddress;
            try
            {
                if (IPAddress.TryParse(target, out targetAsIpAddress))
                {
                    return true;
                }
            }
            catch
            {
            }
            return false;
        }

        public static TargetInformationFromTargetManager FindSingleTarget(string target, bool printErrorIfFailed, bool verbose)
        {
            var targetInfo = new TargetInformationFromTargetManager();

            // Try to parse "target" string as an IPAddress object.
            // If it is a valid IP address, then append an anchor so that we search for the exact IP.
            // This is necessary because if the last octet is 1 or 2 digits and there is no anchor,
            // the Regex.IsMatch call may match more results than intended.
            IPAddress targetAsIpAddress;
            if (IPAddress.TryParse(target, out targetAsIpAddress))
            {
                target += "$";
            }

            try
            {
                RetryUtility.Do(
                    () =>
                    {
                        var timeout = TimeSpan.FromSeconds(5);
                        var targets = ListTargetCommand.Run(true, timeout, true);

                        if (targets == null)
                        {
                            Console.Error.WriteLine("No target is found.");
                            throw new Exception();
                        }

                        int matchedTargetCount = 0;
                        var matched = Regex.Matches(targets, controlTargetDetectMatcher);
                        foreach (Match i in matched)
                        {
                            if (Regex.IsMatch(i.Groups["ipAddress"].Value, target) ||
                                Regex.IsMatch(i.Groups["name"].Value, target) ||
                                Regex.IsMatch(i.Groups["serialNumber"].Value, target))
                            {
                                targetInfo.name = i.Groups["name"].Value;
                                targetInfo.ipAddress = i.Groups["ipAddress"].Value;
                                targetInfo.tmaVersion = i.Groups["tmaVersion"].Value;
                                targetInfo.macAddress = i.Groups["macAddress"].Value;
                                targetInfo.spec = i.Groups["spec"].Value;
                                targetInfo.connectionType = i.Groups["connectionType"].Value;
                                targetInfo.hardwareType = i.Groups["hardwareType"].Value;
                                targetInfo.serialNumber = i.Groups["serialNumber"].Value;
                                matchedTargetCount++;

                                if (verbose)
                                {
                                    Console.Out.WriteLine("Matched target information");
                                    PrintTargetInformation(targetInfo);
                                }

                            }

                        }

                        if (matchedTargetCount == 0)
                        {
                            throw new Exception("The specified target is not found.");
                        }
                        else if (matchedTargetCount > 1)
                        {
                            if (printErrorIfFailed)
                            {
                                Console.Out.WriteLine();
                                Console.Error.WriteLine("Multiple targets are found. Please specify only one target.");
                            }
                        }
                    },
                    e =>
                    {
                        if (printErrorIfFailed)
                        {
                            Console.Error.WriteLine();
                            Console.Error.WriteLine(e.Message);
                        }
                    },
                    3,
                    TimeSpan.FromSeconds(5));
            }
            catch
            {
                targetInfo = null;
            }
            return targetInfo;
        }

        public static TargetInformationFromTargetManager FindSingleTargetFromMac(string targetMac, bool verbose)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            int matchedTargetCount = 0;
            foreach (var element in infos)
            {
                if (element.macAddress == targetMac)
                {
                    if (verbose)
                    {
                        Console.Out.WriteLine("Matched target information");
                        PrintTargetInformation(element);
                    }
                    targetInfo = element;
                    matchedTargetCount++;
                }
            }

            if (matchedTargetCount == 0)
            {
                return null;
            }
            else if (matchedTargetCount > 1)
            {
                // ここに来ることはない
                return null;
            }

            return targetInfo;
        }

        public static TargetInformationFromTargetManager FindSingleTargetFromIp(string targetIp, bool verbose)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            int matchedTargetCount = 0;
            foreach (var element in infos)
            {
                if (element.ipAddress == targetIp)
                {
                    if (verbose)
                    {
                        Console.Out.WriteLine("Matched target information");
                        PrintTargetInformation(element);
                    }
                    targetInfo = element;
                    matchedTargetCount++;
                }
            }

            if (matchedTargetCount == 0)
            {
                return null;
            }
            else if (matchedTargetCount > 1)
            {
                // ここに来ることはない
                return null;
            }

            return targetInfo;
        }

        public static ExitStatus ListDetectedTarget()
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            Console.Out.WriteLine("Name, IP Address, AgentVersion, MAC Address, Spec, ConnectionType, HardwareType, Serial Number");
            if( infos == null)
            {
                return ExitStatus.Success;
            }

            foreach (var t in infos)
            {
                PrintTargetInformation(t);
            }
            return ExitStatus.Success;
        }

        public static String GetTargetMacFromIp(string targetArgumentString)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            foreach (var element in infos)
            {
                if (element.ipAddress == targetArgumentString)
                {
                    return element.macAddress;
                }
            }
            return null;
        }

        public static String GetTargetMacFromName(string targetArgumentString)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            foreach (var element in infos)
            {
                if (element.name == targetArgumentString)
                {
                    return element.macAddress;
                }
            }
            return null;
        }

        public static String GetTargetMacFromSerial(string targetArgumentString)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            foreach (var element in infos)
            {
                if (element.serialNumber == targetArgumentString)
                {
                    return element.macAddress;
                }
                else if (element.serialNumber.ToUpper() == targetArgumentString)
                {
                    Console.Error.WriteLine("[WARN] ***********************************************************");
                    Console.Error.WriteLine("[WARN]  The serial number may be case different. ({0})", element.serialNumber);
                    Console.Error.WriteLine("[WARN]  Please contact to Nintendo support office.");
                    Console.Error.WriteLine("[WARN] ***********************************************************");
                }
            }
            return null;
        }

        public static String GetTargetIpFromMac(string targetArgumentString)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            foreach (var element in infos)
            {
                if (element.macAddress == targetArgumentString)
                {
                    return element.ipAddress;
                }
            }
            return null;
        }

        public static String GetTargetSerialFromMac(string targetArgumentString)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            foreach (var element in infos)
            {
                if (element.macAddress == targetArgumentString)
                {
                    return element.serialNumber;
                }
            }
            return null;
        }

        public static String GetTargetSerialFromIp(string targetArgumentString)
        {
            TargetInformationFromTargetManager[] infos = GetTargetInformations();
            if (infos == null)
            {
                Console.Out.WriteLine();
                Console.Error.WriteLine("The specified target is not found.");
                return null;
            }

            var targetInfo = new TargetInformationFromTargetManager();
            foreach (var element in infos)
            {
                if (element.ipAddress == targetArgumentString)
                {
                    return element.serialNumber;
                }
            }
            return null;
        }

    }
}
