﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.IO.Ports;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;

namespace Nintendo.ControlTarget
{
    public class SerialPortWrapper : IDisposable
    {
        public SerialPortWrapper()
        {
            RawPort = OpenSerialPort(GetFirstPortName());
            StartDataflow();
        }

        public SerialPortWrapper(string portName)
        {
            RawPort = OpenSerialPort(portName);
            StartDataflow();
        }

        public SerialPortWrapper(string portName, int baudRate, Parity parity, int dataBits, StopBits stopBits)
        {
            RawPort = OpenSerialPort(portName, baudRate, parity, dataBits, stopBits);
            StartDataflow();
        }

        public SerialPortWrapper(SerialPort port)
        {
            RawPort = port;
            StartDataflow();
        }

        public static bool IsAvailableFirstPort()
        {
            return 0 < GetPortNames().Length;
        }

        public static string GetFirstPortName()
        {
            if (!IsAvailableFirstPort())
            {
                throw new InvalidUsage("Found no available serial port");
            }

            return GetPortNames()[0];
        }

        public static string[] GetPortNames()
        {
            return SerialPort.GetPortNames();
        }

        public string SendCommand(string command)
        {
            var resultBlock = new BufferBlock<string>();

            using (var unlink = BroadcastBlock.LinkTo(resultBlock))
            {
                return SendCommand(command, WritePort, resultBlock, TimeSpan.FromMilliseconds(500));
            }
        }

        public SerialPort RawPort { get; private set; }

        public ISourceBlock<string> ReadPort { get { return ReadBuffer; } }
        public ITargetBlock<string> WritePort { get { return WriteBuffer; } }

        private BroadcastBlock<string> BroadcastBlock = new BroadcastBlock<string>(s => { return s; });
        private BufferBlock<string> ReadBuffer = new BufferBlock<string>();
        private BufferBlock<string> WriteBuffer = new BufferBlock<string>();

        private CancellationTokenSource cancelSource = new CancellationTokenSource();

        private Task ReadTask;
        private Task WriteTask;

        private void StartDataflow()
        {
            BroadcastBlock.LinkTo(ReadBuffer);

            ReadTask = Task.Factory.StartNew(() =>
            {
                try
                {
                    var readStream = new StreamReader(RawPort.BaseStream);
                    while (RawPort.IsOpen)
                    {
                        const int BUFFER_SIZE = 4096;
                        char[] buffer = new char[BUFFER_SIZE];

                        var read = readStream.Read(buffer, 0, BUFFER_SIZE);

                        BroadcastBlock.Post(new string(buffer, 0, read));
                    }
                }
                catch (IOException)
                {
                }
            });

            WriteTask = Task.Factory.StartNew(async () =>
            {
                while (await WriteBuffer.OutputAvailableAsync())
                {
                    RawPort.Write(WriteBuffer.Receive());
                }
            });
        }

        public static string SendCommand(string command, ITargetBlock<string> target, ISourceBlock<string> source, TimeSpan timeout)
        {
            var builder = new StringBuilder();
            var receiveTask = Task.Factory.StartNew(() =>
            {
                BufferBlock<string> resultBlock = new BufferBlock<string>();
                while (true)
                {
                    builder.Append(source.Receive());
                }
            });
            target.Post(command + "\n");

            receiveTask.Wait(timeout);
            return builder.ToString();
        }

        private static SerialPort OpenSerialPort(string portName)
        {
            return OpenSerialPort(portName, 115200, Parity.None, 8, StopBits.One);
        }

        private static SerialPort OpenSerialPort(string portName, int baudRate, Parity parity, int dataBits, StopBits stopBits)
        {
            Trace.WriteLine("Connect serial port: {0}", portName);

            var serialPort = new SerialPort(portName, baudRate, parity, dataBits, stopBits);

            serialPort.DtrEnable = true;
            serialPort.RtsEnable = true;

            RetryUtility.Do(
                () =>
                {
                    Trace.WriteLine("Try to connect serial port: {0}", portName);
                    serialPort.Open();
                },
                e =>
                {
                    Trace.WriteLine("Failure to connect serial port. Reason: {0}", e.ToString());
                },
                2,
                TimeSpan.FromSeconds(1));

            Trace.WriteLine("Connected: {0}", portName);

            return serialPort;
        }

        public void Dispose()
        {
            RawPort.Dispose();
        }
    }
}
