﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;

namespace Nintendo.ControlTarget
{
    public class DataflowUtility
    {
        public static IPropagatorBlock<string, Tuple<string, string>> CreateAdditionMetaBlock(string meta)
        {
            return new TransformBlock<string, Tuple<string, string>>(s => { return new Tuple<string, string>(meta, s); });
        }

        public static IPropagatorBlock<Tuple<string, string>, string> CreateDelitionMetaBlock()
        {
            return new TransformBlock<Tuple<string, string>, string>(t => { return t.Item2; });
        }

        public static IPropagatorBlock<Tuple<string, string>, string> CreateArrangementMultiLogBlock()
        {
            var inputBuffer = new BufferBlock<Tuple<string, string>>();
            var outputBuffer = new BufferBlock<string>();

            string previous = null;
            var lastHalfwayLine = new Dictionary<string, string>();

            var arrangeAction = new ActionBlock<Tuple<string, string>>(logWithMeta =>
            {
                var lines = logWithMeta.Item2.Split(new string[] { "\r\n", "\n" }, StringSplitOptions.None).ToList().ConvertAll(s => s + "\r\n");
                if (0 < lines.Count && lines.LastOrDefault() == "\r\n")
                {
                    lines.RemoveAt(lines.Count - 1);
                }
                else
                {
                    lines[lines.Count - 1] = lines[lines.Count - 1].Remove(lines[lines.Count - 1].Length - 2);
                }

                foreach (var line in lines)
                {
                    var t = new Tuple<string, string>(logWithMeta.Item1, line);

                    if (previous == t.Item1)
                    {
                        lastHalfwayLine[t.Item1] = lastHalfwayLine[t.Item1] + t.Item2;
                        outputBuffer.Post(t.Item2);
                    }
                    else
                    {
                        if (previous != null)
                        {
                            outputBuffer.Post("\r\n");
                        }

                        if (lastHalfwayLine.ContainsKey(t.Item1))
                        {
                            lastHalfwayLine[t.Item1] = lastHalfwayLine[t.Item1] + t.Item2;
                        }
                        else
                        {
                            var header = t.Item1 != string.Empty ? string.Format("[{0}] ", t.Item1) : string.Empty;
                            lastHalfwayLine[t.Item1] = string.Format("{0}{1}", header, t.Item2);
                        }
                        outputBuffer.Post(lastHalfwayLine[t.Item1]);
                    }
                    previous = t.Item1;

                    if (t.Item2.LastOrDefault() == '\n')
                    {
                        lastHalfwayLine.Remove(t.Item1);
                        previous = null;
                    }
                    else
                    {
                        previous = t.Item1;
                    }
                }
            });

            var block = DataflowBlock.Encapsulate(inputBuffer, outputBuffer);

            DataflowUtility.LinkBlock("inputBuffer->arrangeAction", inputBuffer, arrangeAction);
            DataflowUtility.AddLinkBlockWithoutLink("block->inputBuffer", block, inputBuffer);
            DataflowUtility.AddLinkBlockWithoutLink("block->outputBuffer", inputBuffer, outputBuffer);

            return block;
        }

        public static IPropagatorBlock<string, string> CreateHalfwayLineSplitter()
        {
            string rest = string.Empty;

            var source = new BufferBlock<string>();

            var target = new ActionBlock<string>(text =>
            {
                text = rest + text.Replace("\r", string.Empty);

                var splitted = text.Split('\n');

                foreach (var line in text.Split('\n'))
                {
                    if (line != string.Empty)
                    {
                        source.Post(line);
                    }
                }

                if (0 < splitted.Length)
                {
                    rest = splitted[splitted.Length - 1];
                }
            });

            var block = DataflowBlock.Encapsulate(target, source);

            DataflowUtility.AddLinkBlockWithoutLink(string.Format("HalfwayLineSplitter::block->source {0}", source.GetHashCode()), block, source);
            DataflowUtility.AddLinkBlockWithoutLink(string.Format("HalfwayLineSplitter::source->target {0}", source.GetHashCode()), source, target);

            return block;
        }

        public static ITargetBlock<string> CreateStandardOutputTarget()
        {
            var target = new ActionBlock<string>(s =>
            {
                Console.Write(s.Replace("\0", ""));
            });

            return target;
        }

        public static ITargetBlock<string> CreateTextOutputTarget(FileInfo output)
        {
            var file = new StreamWriter(output.FullName);

            var ret = new ActionBlock<string>(s =>
            {
                file.Write(s);
            });

            ret.Completion.ContinueWith(t =>
            {
                file.Dispose();
            }, TaskContinuationOptions.ExecuteSynchronously);

            return ret;
        }

        public static ITargetBlock<string> CreateTargetBlockFromOutputOption(string outputPath, ITargetBlock<string> defaultTarget)
        {
            if (outputPath == null)
            {
                return defaultTarget;
            }
            else
            {
                return CreateTextOutputTarget(new FileInfo(outputPath));
            }
        }

        public static ITargetBlock<Tuple<string, string>> CreateTargetBlockFromOutputOption(string outputPath, ITargetBlock<Tuple<string, string>> defaultTarget)
        {
            if (outputPath == null)
            {
                return defaultTarget;
            }
            else
            {
                var delitionTarget = DataflowUtility.CreateDelitionMetaBlock();
                var fileTarget = DataflowUtility.CreateTextOutputTarget(new FileInfo(outputPath));
                DataflowUtility.LinkBlock("delitionTarget->fileTarget", delitionTarget, fileTarget);
                return delitionTarget;
            }
        }

        public static IDisposable LinkBlock<T>(string linkName, ISourceBlock<T> source, ITargetBlock<T> target)
        {
            AddLinkBlockWithoutLink(linkName, source, target);

            return source.LinkTo(target, new DataflowLinkOptions() { PropagateCompletion = true });
        }

        public static void AddLinkBlockWithoutLink<T1, T2>(string linkName, ISourceBlock<T1> source, ITargetBlock<T2> target)
        {
            DataflowBlockContainer.Add(source);
            DataflowBlockContainer.Add(target);
            LinkContainer.Add(string.Format("{0}: {1}->{2}", linkName, source.GetHashCode(), target.GetHashCode()), new Tuple<IDataflowBlock, IDataflowBlock>(source, target));

            BlockLinks.Add(new Tuple<IDataflowBlock, IDataflowBlock>(source, target));

            source.Completion.ContinueWith(t =>
            {
                target.Complete();
                target.Completion.Wait();
            }, TaskContinuationOptions.ExecuteSynchronously);
        }



        public static Dictionary<string, Tuple<IDataflowBlock, IDataflowBlock>> LinkContainer = new Dictionary<string, Tuple<IDataflowBlock, IDataflowBlock>>();
        public static List<Tuple<IDataflowBlock, IDataflowBlock>> BlockLinks = new List<Tuple<IDataflowBlock, IDataflowBlock>>();
        public static HashSet<IDataflowBlock> DataflowBlockContainer = new HashSet<IDataflowBlock>();
    }
}
