﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.Diagnostics;

namespace Nintendo.ControlTarget
{
    public class Disposable : IDisposable
    {
        public Disposable(Action action)
        {
            this.Action = action;
        }

        public void Dispose()
        {
            this.Action();
        }

        private Action Action;
    }

    public class RunnerToolCoordinator
    {
        public RunnerToolCoordinator(string targetName)
        {
            this.InterruptEvent = new EventWaitHandle(false, EventResetMode.AutoReset, "RunnerToolCoordinator:InterruptEvent:" + targetName);
            this.OwnedMutex = new Mutex(false, "RunnerToolCoordinator:OwnedMutex:" + targetName);
        }

        public void Interrupt()
        {
            this.InterruptEvent.Set();
        }

        public IDisposable RequestOwnership(int retryCount = 5)
        {
            RetryUtility.Do(
                () =>
                {
                    Trace.WriteLine("Request to ownership");
                    Owned = OwnedMutex.WaitOne(TimeSpan.FromSeconds(1));

                    if (!Owned)
                    {
                        throw new Exception("Failed to get ownership");
                    }
                },
                (e) =>
                {
                    Trace.WriteLine("Interrupt RunnerTools");
                    Interrupt();
                },
                (es) =>
                {
                    throw new Exception("Failed to get ownership");
                },
                retryCount,
                TimeSpan.FromMilliseconds(1000));

            InterruptEvent.Reset();

            ForceExitThread = new Thread(() =>
            {
                while (!(InterruptEvent.WaitOne(TimeSpan.FromMilliseconds(100))))
                {
                    Thread.Sleep(TimeSpan.FromSeconds(1));
                }
                Thread.Sleep(TimeSpan.FromSeconds(3));

                Trace.WriteLine("Interrupted by other process. Exit forcibly.");
                System.Environment.Exit(1);
            });

            ForceExitThread.Start();

            return new Disposable(() => { this.ReleaseOwnership(); });
        }

        public void ReleaseOwnership()
        {
            ForceExitThread.Abort();
            InterruptEvent.Reset();
            OwnedMutex.ReleaseMutex();
        }

        public async Task WaitInterrupt(CancellationToken cancelToken)
        {
            while ( !(InterruptEvent.WaitOne(TimeSpan.FromMilliseconds(100)) || cancelToken.IsCancellationRequested) )
            {
                await Task.Delay(100);
            }
        }

        public WaitHandle WaitHandle { get { return this.InterruptEvent; } }

        private EventWaitHandle InterruptEvent;
        private Mutex OwnedMutex;
        private bool Owned = false;
        private Thread ForceExitThread;
    }
}
