﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Net;
using System.Net.Sockets;
using System.Diagnostics;
using System.IO;

namespace Nintendo.ControlTarget
{
    public class HostBridgeAccessor : IDisposable
    {
        public HostBridgeAccessor(IPEndPoint endPoint)
        {
            Initialize(endPoint);
        }

        private void Initialize(IPEndPoint endPoint)
        {
            Trace.WriteLine($"Connect: {endPoint}");

            try
            {
                RetryUtility.Do(
                    () =>
                    {
                        Trace.WriteLine($"Try to connect: {endPoint}");
                        socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
                        var result = socket.BeginConnect(endPoint, null, null);
                        if (!result.AsyncWaitHandle.WaitOne(1000, true))
                        {
                            socket.EndConnect(result);
                            throw new TimeoutException();
                        }

                        if (!socket.Connected)
                        {
                            throw new Exception("Not connected");
                        }
                    },
                    e =>
                    {
                        Trace.WriteLine($"[ERROR] Failed to connect: {e.Message}");
                    },
                    es =>
                    {
                        Console.Error.WriteLine($"[ERROR] Failed to connect: {es.Last().Message}");
                    },
                    20,
                    TimeSpan.FromSeconds(1));
            }
            catch (Exception e)
            {
                throw new Exception($"Failed to connect: {e.Message}");
            }

            socketStream = new NetworkStream(socket);
            socketWriter = new StreamWriter(socketStream);
            socketWriter.AutoFlush = true;
            socketReader = new StreamReader(socketStream);

            Trace.WriteLine($"Connected: {endPoint}");

            StartDataflow();
        }

        private void StartDataflow()
        {
            DataflowUtility.LinkBlock("BroadCastBlock->NullBlock", BroadCastBlock, NullBlock);
            BroadCastBlock.Post(string.Empty);

            ReadTask = Task.Factory.StartNew(() =>
            {
                try
                {
                    while (true)
                    {
                        const int BUFFER_SIZE = 4096;
                        byte[] buffer = new byte[BUFFER_SIZE];
                        byte[] eliminatedBuffer1 = new byte[BUFFER_SIZE];
                        byte[] eliminatedBuffer2 = new byte[BUFFER_SIZE];
                        var read = socketStream.Read(buffer, 0, BUFFER_SIZE);
                        var eliminatedRead1 = eliminator.CopyWithElimination(eliminatedBuffer1, buffer, read);
                        var eliminatedRead2 = escapeSequenceEliminator.CopyWithElimination(eliminatedBuffer2, eliminatedBuffer1, read);
                        var eliminatedString = Encoding.UTF8.GetString(eliminatedBuffer2, 0, eliminatedRead2);
                        BroadCastBlock.Post(eliminatedString);
                        var rawString = Encoding.UTF8.GetString(buffer, 0, read);
                        RawBroadCastBlock.Post(rawString);
                    }
                }
                catch (IOException)
                {
                }
                catch (ObjectDisposedException)
                {
                }
            });

            WriteTask = Task.Factory.StartNew(async () =>
            {
                while (await WriteBuffer.OutputAvailableAsync())
                {
                    var text = WriteBuffer.Receive();
                    socketWriter.Write(text);
                }
            }).Unwrap();
        }

        public void Dispose()
        {
            Trace.WriteLine("Disconnect the socket.");
            socket.Disconnect(false);
            socketWriter.Dispose();
            socketStream.Dispose();
            socket.Dispose();
            Trace.WriteLine("Disposed the socket.");
        }

        public ISourceBlock<string> ReadPort { get { return BroadCastBlock; } }

        public ISourceBlock<string> RawReadPort { get { return RawBroadCastBlock; } }
        public ITargetBlock<string> WritePort { get { return WriteBuffer; } }

        public const int DefaultPort = 10023;

        private BroadcastBlock<string> BroadCastBlock = new BroadcastBlock<string>(s => { return s; });
        private BroadcastBlock<string> RawBroadCastBlock = new BroadcastBlock<string>(s => { return s; });
        private BufferBlock<string> WriteBuffer = new BufferBlock<string>();
        private ITargetBlock<string> NullBlock = DataflowBlock.NullTarget<string>();
        private Task ReadTask;
        private Task WriteTask;
        private Socket socket;
        private NetworkStream socketStream;
        private StreamWriter socketWriter;
        private StreamReader socketReader;
        private TelnetIacEliminator eliminator = new TelnetIacEliminator();
        private EscapeSequenceEliminator escapeSequenceEliminator = new EscapeSequenceEliminator();
    }
}
