﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.IO;

namespace Nintendo.ControlTarget.ConsoleShell
{
    public class ConsoleShellResponseId
    {
        public const int Success = 1;
        public const int Failure = 2;
        public const int ApplicationExit = 3;
        public const int JitDebugNotification = 5;
        public const int ScreenShot = 8;
    }

    public class CommandResponseHeader
    {
        public const int HeaderSize = 16;
        public UInt64 RequestId { get; private set; }
        public Int32 CommandId { get; private set; }
        public UInt32 BodySize { get; private set; }

        public CommandResponseHeader(UInt64 requestId, Int32 commandId, UInt32 bodySize)
        {
            RequestId = requestId;
            CommandId = commandId;
            BodySize = bodySize;
        }

        public CommandResponseHeader(byte[] responseBinary)
        {
            using (var memoryStream = new MemoryStream(responseBinary))
            using (var binaryReader = new BinaryReader(memoryStream))
            {
                RequestId = binaryReader.ReadUInt64();
                CommandId = binaryReader.ReadInt32();
                BodySize = binaryReader.ReadUInt32();
            }
        }
    }

    public abstract class IConsoleShellResponse
    {
        public CommandResponseHeader Header { get; private set; }

        public IConsoleShellResponse(CommandResponseHeader header)
        {
            Header = header;
        }

        public IConsoleShellResponse(CommandResponseHeader header, byte[] body)
        {
            Header = header;

            using (var memoryStream = new MemoryStream(body))
            using (var binaryReader = new BinaryReader(memoryStream))
            {
                this.ConstructObject(binaryReader);
            }
        }

        public byte[] ConstructResponseData()
        {
            using (var memoryStream = new MemoryStream())
            using (var binaryWriter = new BinaryWriter(memoryStream))
            {
                binaryWriter.Write(Header.RequestId);
                binaryWriter.Write(Header.CommandId);
                binaryWriter.Write(Header.BodySize);
                this.ConstructBody(binaryWriter);

                return memoryStream.ToArray();
            }
        }

        public bool Match<ResponseType>()
            where ResponseType : IConsoleShellResponse
        {
            return this.GetType() == typeof(ResponseType);
        }

        public bool ProcessIf<ResponseType>(Action<ResponseType> processor)
            where ResponseType : IConsoleShellResponse
        {
            if (this.GetType() == typeof(ResponseType))
            {
                processor((ResponseType)this);
                return true;
            }

            return false;
        }

        public abstract void ConstructObject(BinaryReader reader);
        public abstract void ConstructBody(BinaryWriter writer);

        public static IConsoleShellResponse MakeResponse(CommandResponseHeader header, byte[] body)
        {
            switch (header.CommandId)
            {
                case ConsoleShellResponseId.Success:
                    return new SuccessResponse(header, body);
                case ConsoleShellResponseId.Failure:
                    return new FailureResponse(header, body);
                case ConsoleShellResponseId.ApplicationExit:
                    return new ApplicationExitResponse(header, body);
                case ConsoleShellResponseId.JitDebugNotification:
                    return new JitDebugNotificationResponse(header, body);
                case ConsoleShellResponseId.ScreenShot:
                    return new ScreenShotResponse(header, body);
                default:
                    throw new Exception(string.Format("Unknown console shell command: id = {0}", header.CommandId));
            }
        }

        public abstract string ResponseExpression { get; }
    }

    public class SuccessResponse : IConsoleShellResponse
    {
        public SuccessResponse(ulong requestId)
            : base(new CommandResponseHeader(requestId, ConsoleShellResponseId.Success, 0))
        {
        }

        public SuccessResponse(CommandResponseHeader header, byte[] body)
            : base(header, body)
        {
        }

        public override void ConstructObject(BinaryReader reader)
        {
        }

        public override void ConstructBody(BinaryWriter writer)
        {
        }

        public override string ResponseExpression
        {
            get
            {
                return string.Format("SuccessResponse(id:0x{0,0:X8})", Header.RequestId);
            }
        }
    }

    public class FailureResponse : IConsoleShellResponse
    {
        public UInt32 Result { get; private set; }

        public FailureResponse(ulong requestId, uint result)
            : base(new CommandResponseHeader(requestId, ConsoleShellResponseId.Failure, 8))
        {
            Result = result;
        }

        public FailureResponse(CommandResponseHeader header, byte[] body)
            : base(header, body)
        {
        }

        public override void ConstructObject(BinaryReader reader)
        {
            Result = reader.ReadUInt32();
            reader.ReadUInt32();
        }

        public override void ConstructBody(BinaryWriter writer)
        {
            writer.Write(Result);
            writer.Write((uint)0);
        }

        public override string ResponseExpression
        {
            get
            {
                return string.Format("FailureResponse(id:0x{0,0:X8}, result:0x{1,0:X8})", Header.RequestId, this.Result);
            }
        }
    }

    public class ApplicationExitResponse : IConsoleShellResponse
    {
        public int ExitStatus;

        public ApplicationExitResponse(ulong requestId, int exitStatus)
            : base(new CommandResponseHeader(requestId, ConsoleShellResponseId.ApplicationExit, 0))
        {
            ExitStatus = exitStatus;
        }

        public ApplicationExitResponse(CommandResponseHeader header, byte[] body)
            : base(header, body)
        {
        }

        public override void ConstructObject(BinaryReader reader)
        {
        }

        public override void ConstructBody(BinaryWriter writer)
        {
            ExitStatus = 0;
        }

        public override string ResponseExpression
        {
            get
            {
                return string.Format("ApplicationExitResponse(id:0x{0,0:X8})", Header.RequestId);
            }
        }
    }

    public class JitDebugNotificationResponse : IConsoleShellResponse
    {
        public int ExitStatus;

        public JitDebugNotificationResponse(ulong requestId, int exitStatus)
            : base(new CommandResponseHeader(requestId, ConsoleShellResponseId.JitDebugNotification, 0))
        {
            ExitStatus = exitStatus;
        }

        public JitDebugNotificationResponse(CommandResponseHeader header, byte[] body)
            : base(header, body)
        {
            ExitStatus = 1;
        }

        public override void ConstructObject(BinaryReader reader)
        {
        }

        public override void ConstructBody(BinaryWriter writer)
        {
        }

        public override string ResponseExpression
        {
            get
            {
                return string.Format("JitNotificationResponse(id:0x{0,0:X8})", Header.RequestId);
            }
        }
    }

    public class ScreenShotResponse : IConsoleShellResponse
    {
        private byte[] Data;
        public ScreenShotResponse(ulong requestId, int exitStatus)
            : base(new CommandResponseHeader(requestId, 8, 0))
        {
        }

        public ScreenShotResponse(CommandResponseHeader header, byte[] body)
            : base(header, body)
        {
        }

        public override void ConstructObject(BinaryReader reader)
        {
            Data = reader.ReadBytes((int)Header.BodySize);
        }

        public override void ConstructBody(BinaryWriter writer)
        {
        }

        public override string ResponseExpression
        {
            get
            {
                return string.Format("SSResponse(id:0x{0,0:X8}, size {1})", Header.RequestId, Header.BodySize);
            }
        }

        public void Save(string path)
        {
            try
            {
                FileStream fs = new FileStream(path, FileMode.Create);
                fs.Write(Data, 0, (int)Header.BodySize);
                fs.Close();
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.ToString());
            }
        }
    }
}
