﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Linq;
using Nintendo.Foundation.IO;

namespace Nintendo.ControlTarget
{
    public class ListTargetCommandArgument : CommandCommon
    {
        [CommandLineOption("xml",
            Description = "Output target infomations by XML.",
            DefaultValue = false)]
        public bool Xml { get; set; }

        [CommandLineOption("csv",
            Description = "Output target infomations by CSV.",
            DefaultValue = false)]
        public bool Csv { get; set; }

        public override void Run()
        {
            RunCommand(
                () =>
                {
                    if (Xml)
                    {
                        Console.WriteLine(ListTargetCommand.ListTargetAsXml());
                    }
                    else if (Csv)
                    {
                        Console.Write(ListTargetCommand.ListTargetAsCsv());
                    }
                    else
                    {
                        Console.WriteLine(ListTargetCommand.Run(false, TimeSpan.FromSeconds(0), false));
                    }
                });
        }
    }

    public class ListTargetCommand
    {
        public static string Run()
        {
            using (var accessor = new TargetManagerAccessor())
            {
                accessor.EnsureStart();

                var targets = accessor.ListTarget();

                var ret = new StringBuilder();

                foreach (var target in targets)
                {
                    ret.AppendLine(string.Format("{0}\t{1}\t{2}",
                        string.IsNullOrEmpty(target.PeerType) ? "<NoPeerType>" : target.PeerType,
                        string.IsNullOrEmpty(target.PeerName) ? "<NoPeerName>" : target.PeerName,
                        string.IsNullOrEmpty(target.Id) ? "<NoId>" : target.Id));
                }

                return ret.ToString().TrimEnd();
            }
        }

        public static string Run(bool detect, TimeSpan timeout, bool allInfo)
        {
            using (var accessor = new TargetManagerAccessor())
            {
                accessor.EnsureStart();

                var ret = new StringBuilder();

                if (detect)
                {
                    var targets = accessor.ListDetectedTarget(timeout);

                    foreach (var target in targets)
                    {
                        if (allInfo)
                        {
                            ret.AppendLine(string.Format("{0}\t{1}\t{2}\t{3}\t{4}\t{5}\t{6}",
                                string.IsNullOrEmpty(target.GetName()) ? "<NoName>" : target.GetName(),
                                string.IsNullOrEmpty(target.GetIpAddress()) ? "<NoIpAddress>" : target.GetIpAddress(),
                                string.IsNullOrEmpty(target.GetTmaVersion()) ? "<NoTmaVersion>" : target.GetTmaVersion(),
                                string.IsNullOrEmpty(target.GetMacAddress()) ? "<NoMacAddress>" : target.GetMacAddress(),
                                string.IsNullOrEmpty(target.GetConnectionType()) ? "<NoConnectionType>" : target.GetConnectionType(),
                                string.IsNullOrEmpty(target.GetHardwareType()) ? "<NoHardwareType>" : target.GetHardwareType(),
                                string.IsNullOrEmpty(target.GetSerialNumber()) ? "<NoSerialNumber>" : target.GetSerialNumber()
                                ));
                        }
                        else
                        {
                            ret.AppendLine(string.Format("{0}\t{1}",
                                string.IsNullOrEmpty(target.GetName()) ? "<NoName>" : target.GetName(),
                                string.IsNullOrEmpty(target.GetIpAddress()) ? "<NoIpAddress>" : target.GetIpAddress()));
                        }
                    }
                }
                else
                {
                    var targets = accessor.ListRegisteredTarget();

                    foreach (var target in targets)
                    {
                        ret.AppendLine(GetTargetInfoText(target));
                    }
                }

                return ret.ToString().TrimEnd();
            }
        }

        private static string GetTargetInfoText(Tmapi.RegisteredTargetInfo target)
        {
            return string.Format("{0}\t{1}",
                string.IsNullOrEmpty(target.GetName()) ? "<NoName>" : target.GetName(),
                string.IsNullOrEmpty(target.GetIpAddress()) ? "<NoIpAddress>" : target.GetIpAddress());
        }

        public static string ListTargetAsXml()
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return TargetManagerAccessor.GetRegisteredTargetInfosAsXml(tma.ListRegisteredTarget());
            }
        }

        public static string DetectTargetAsXml(TimeSpan timeout)
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return TargetManagerAccessor.GetTargetInfosAsXml(tma.ListDetectedTarget(timeout));
            }
        }

        public static string ListTargetAsCsv()
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return TargetManagerAccessor.GetRegisteredTargetInfosAsCsv(tma.ListRegisteredTarget());
            }
        }

        public static string DetectTargetAsCsv(TimeSpan timeout)
        {
            using (var tma = new TargetManagerAccessor())
            {
                tma.EnsureStart();

                return TargetManagerAccessor.GetTargetInfosAsCsv(tma.ListDetectedTarget(timeout));
            }
        }

    }
}
