﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Reflection;
using Nintendo.Foundation.IO;
using CommandUtility;

namespace Nintendo.ControlTarget
{
    public abstract class CommandCommon
    {
        [CommandLineOption('v', "verbose",
        Description = "Output trace logs.")]
        public bool Verbose { get; set; }

        public abstract void Run();

        public void RunCommand(Action action)
        {
            DataflowConsole.Instance.SetOption(this.Verbose);
            ConsoleApplicationTraceListener.SetGlobal(this.Verbose);

            action();
        }

        public static CommandCommon FindActivatedSubCommand(object commandContainer)
        {
            var containerType = commandContainer.GetType();
            foreach (var property in containerType.GetProperties())
            {
                if (0 < property.GetCustomAttributes<CommandLineSubCommandAttribute>(true).Count())
                {
                    var subCommandValue = property.GetValue(commandContainer);
                    if (subCommandValue is CommandCommon && subCommandValue != null)
                    {
                        return (CommandCommon)subCommandValue;
                    }
                }
            }

            throw new Exception("Found no subcommands");
        }
    }

    public abstract class CommonTargetManagerCommand : CommandCommon
    {
        public void RunTargetManagerCommand(Action<TargetManagerAccessor> action)
        {
            RunCommand(
                () =>
                {
                    using (var tmapiAccessor = new TargetManagerAccessor())
                    {
                        tmapiAccessor.EnsureStart();

                        action(tmapiAccessor);
                    }
                });
        }
    }

    public abstract class CommonTargetCommand : CommandCommon
    {
        [CommandLineOption('t', "target",
            Description = "Set target name to run the program. (See targets with 'ControlTarget.exe list-target')")]
        public string Target { get; set; }

        public void RunHostBridgeTargetCommand(Action<string, Tmapi.RegisteredTargetInfo, Tmapi.TargetAccessor, TargetManagerAccessor> action)
        {
            RunCommand(
                () =>
                {
                    using (var tmapiAccessor = new TargetManagerAccessor())
                    {
                        tmapiAccessor.EnsureStart();

                        Trace.WriteLine($"Try find target. name={this.Target ?? "DefaultTarget"}");

                        var targetInfo = tmapiAccessor.FindTargetInfo(this.Target);

                        Trace.WriteLine($"Get target accessor. name={this.Target}, handle={targetInfo.GetTargetHandle()}");

                        using (var target = tmapiAccessor.GetTarget(targetInfo.GetTargetHandle()))
                        {
                            action(this.Target, targetInfo, target, tmapiAccessor);
                        }
                    }
                });
        }
    }

    public abstract class CommonDevMenuCommandCommand : CommonTargetCommand
    {
        [CommandLineValues(ValueName = "DevMenuCommand Arguments", Description = "Set arguments for DevMenuCommand.")]
        public string[] Arguments { get; set; } = new string[] { };

        [CommandLineOption('d', "devmenucommand",
            Description = "Set path to DevMenuCommand.")]
        public string DevMenuCommandPath { get; set; }

        [CommandLineOption('r', "runontarget",
            Description = "Set path to RunOnTarget.exe .")]
        public string RunOnTargetPath { get; set; }

        public string TargetName { get; private set; }

        public void RunDevMenuCommand(params string[] arguments)
        {
            var wholeArguments = new string[]
            {
                DevMenuCommandPath,
                "-t",
                TargetName,
                "--"
            }.Concat(arguments).Concat(Arguments).ToArray();

            SdkTool.Execute(new FileInfo(RunOnTargetPath), wholeArguments, null, message => Console.WriteLine(message), null);
        }

        public void RunDevMenuCommandCommand(Action action)
        {
            RunHostBridgeTargetCommand(
                (name, info, target, tmapiAccessor) =>
                {
                    TargetName = info.GetSerialNumber();
                    DevMenuCommandPath = GetDevMenuCommandPath(DevMenuCommandPath);
                    RunOnTargetPath = GetRunOnTargetPath(RunOnTargetPath);

                    action();
                }
            );
        }

        private string GetRunOnTargetPath(string runOnTargetPath)
        {
            if (runOnTargetPath == null)
            {
                runOnTargetPath = SdkPath.FindToolPath("RunOnTarget.exe").FullName;
            }

            if (File.Exists(runOnTargetPath))
            {
                return runOnTargetPath;
            }

            throw new Exception("Not found RunOnTarget.exe.");
        }

        private string GetDevMenuCommandPath(string devMenuCommandPath)
        {
            if (devMenuCommandPath == null)
            {
                return "system:0x0100000000002101";
            }

            if (File.Exists(devMenuCommandPath))
            {
                return devMenuCommandPath;
            }

            throw new Exception("Not found DevMenuCommand.");
        }
    }
}
