﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.Net.NetworkInformation;

namespace Nintendo.Bridge
{
    delegate void NotifiedHandler(IPEndPoint endPoint, byte[] info);

    class NetworkAdapter
    {
        private readonly string UDP_BROADCAST_TEXT = "host";

        public event NotifiedHandler notified;

        private IPAddress address;
        private IPAddress netmask;
        private IPAddress broadcast;

        private int port;

        private Thread thread;
        private UdpClient client;

        public bool Verbose { get; set; }

        public NetworkAdapter(UnicastIPAddressInformation info, int port)
        {
            Verbose = false;
            address = info.Address;
            netmask = info.IPv4Mask;
            this.port = port;
        }

        public void Start()
        {
            byte[] mask = netmask.GetAddressBytes();
            byte[] host = address.GetAddressBytes();
            byte[] subnet = new byte[host.Length];

            for (int i = 0; i < subnet.Length; i++)
            {
                subnet[i] = (byte)(host[i] & mask[i]);
                mask[i] = (byte)(~mask[i]);
                subnet[i] |= mask[i];
            }

            broadcast = new IPAddress(subnet);
            IPEndPoint endPoint = new IPEndPoint(address, port);
            try
            {
                client = new UdpClient(endPoint);
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                return;
            }
            client.Client.ReceiveBufferSize = 65536;

            thread = new Thread(listen);
            thread.IsBackground = true;
            thread.Start();
            if (Verbose)
            {
                Console.WriteLine("[{0}] Network Adapter starts listening.", address);
            }
        }

        public void Stop()
        {
            if (client != null)
            {
                client.Close();
            }
        }

        public void LegacyBroadcast(char prefix)
        {
            byte[] keyword = Encoding.UTF8.GetBytes(prefix + "HOTBRIDGE_NETWORK_BOARD");

            IPEndPoint endPoint = new IPEndPoint(IPAddress.Broadcast, port);

            try
            {
                client.Send(keyword, keyword.Length, endPoint);
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
            }
        }

        public void Broadcast(char prefix)
        {
            byte[] magic = BitConverter.GetBytes((UInt16)0xFACE);
            byte version = 1;
            byte flag = Convert.ToByte(prefix);
            byte[] keyword = Encoding.UTF8.GetBytes(UDP_BROADCAST_TEXT);
            byte[] data = new byte[magic.Length+sizeof(byte)+sizeof(byte)+keyword.Length+sizeof(UInt16)+sizeof(byte)*6+sizeof(UInt32)*4];

            magic.CopyTo(data, 0);
            data[magic.Length] = version;
            data[magic.Length + sizeof(byte)] = flag;
            keyword.CopyTo(data, magic.Length + sizeof(byte) + sizeof(byte));

            IPEndPoint endPoint = new IPEndPoint(IPAddress.Broadcast, port);

            try
            {
                client.Send(data, data.Length, endPoint);
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
            }
        }

        private void listen()
        {
            IPEndPoint endPoint = new IPEndPoint(IPAddress.Any, port);

            while (true)
            {
                try
                {
                    byte[] info = client.Receive(ref endPoint);
                    byte[] ip = endPoint.Address.GetAddressBytes();

                    if (notified != null)
                    {
                        if (Verbose)
                        {
                            string debug = string.Format
                                (
                                    "[Recv] : {0}.{1}.{2}.{3}:{4}",
                                    ip[0], ip[1], ip[2], ip[3], endPoint.Port
                                );
                            Console.WriteLine(debug);
                        }

                        notified(endPoint, info);
                    }
                }
                catch (Exception e)
                {
                    if (Verbose)
                    {
                        Console.WriteLine(e);
                    }
                    break;
                }
            }
        }
    }
}
