﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using System.Web.Script.Serialization;

namespace Nn.ResultTool
{
    public class ResultDefinitionSerializer
    {
        public class ResultDefinitions
        {
            public ResultModule[] ResultModules { get; set; }
            public ErrorResult[] ErrorResults { get; set; }
        }

        public class ResultModule
        {
            public string Name { get; set; }
            public int Number { get; set; }
        }

        public class ReadableText
        {
            [XmlAttribute("language")]
            public string Language { get; set; }

            [XmlText]
            public string Text { get; set; }
        }

        public class ErrorResult
        {
            public string Name { get; set; }
            public string Namespace { get; set; }
            public string ModuleName { get; set; }
            public string Package { get; set; }

            [XmlAttribute("abstract")]
            public bool IsAbstract { get; set; }

            [XmlElement(IsNullable = true)]
            public int? DescriptionValue { get; set; }

            public int DescriptionBegin { get; set; }
            public int DescriptionEnd { get; set; }

            [XmlElement]
            public ReadableText[] ReadableText { get; set; }
        }

        public class ReferenceTagInfo
        {
            public string Name { get; set; }
            public string Path { get; set; }
        }

        private static ReadableText[] MakeReadableText(Nn.ResultTool.ResultRange r)
        {
            if (string.IsNullOrEmpty(r.Text))
            {
                return new ReadableText[0];
            }
            else
            {
                return new[] {
                    new ReadableText
                    {
                        Language = null,
                        Text = r.Text,
                    },
                };
            }
        }

        public static ResultDefinitions MakeResultDefinitions(ModuleMap moduleMap, IEnumerable<Nn.ResultTool.ResultRange> results)
        {
            var refferedModuleNames = new HashSet<string>(results.Select(r => r.Module));
            var resultModules = from moduleName in refferedModuleNames
                                let moduleNumber = moduleMap.GetModuleId(moduleName)
                                orderby moduleNumber
                                select new ResultModule
                                {
                                    Name = moduleName,
                                    Number = moduleNumber
                                };
            var errorResults = from r in results
                               orderby r.Module ascending, r.Begin ascending, r.End descending
                               select new ErrorResult
                               {
                                   Name = r.Name,
                                   Namespace = r.Namespace,
                                   ModuleName = r.Module,
                                   IsAbstract = r.Abstract,
                                   Package = r.Package,
                                   DescriptionValue = r.Abstract ? default(int?) : r.Begin,
                                   DescriptionBegin = r.Begin,
                                   DescriptionEnd = r.End,
                                   ReadableText = MakeReadableText(r),
                               };
            return new ResultDefinitions
            {
                ResultModules = resultModules.ToArray(),
                ErrorResults = errorResults.ToArray(),
            };
        }

        public static void WriteAsXml(ModuleMap moduleMap, IEnumerable<Nn.ResultTool.ResultRange> results, System.IO.TextWriter o)
        {
            var resultDefinitions = MakeResultDefinitions(moduleMap, results);
            new XmlSerializer(resultDefinitions.GetType()).Serialize(o, resultDefinitions);
        }

        public static string MakeJsonString(ModuleMap moduleMap, IEnumerable<Nn.ResultTool.ResultRange> results)
        {
            var resultDefinitions = MakeResultDefinitions(moduleMap, results);
            return new JavaScriptSerializer().Serialize(resultDefinitions);
        }

        public static string MakeJsonString(ModuleMap moduleMap, IEnumerable<Nn.ResultTool.ResultRange> results, IReadOnlyDictionary<string, string> refInfos)
        {
            var x = new
            {
                ResultInfos = MakeResultDefinitions(moduleMap, results),
                ReferenceInfos = from e in refInfos
                                 select new ReferenceTagInfo
                                 {
                                     Name = e.Key,
                                     Path = e.Value,
                                 }
            };
            return new JavaScriptSerializer().Serialize(x);
        }
    }
}
