﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace Nn.ResultTool
{
    [Serializable]
    public class ModuleMapException : Exception
    {
        public ModuleMapException() { }
        public ModuleMapException(string message) : base(message) { }
        public ModuleMapException(string message, Exception inner) : base(message, inner) { }
        protected ModuleMapException(
            System.Runtime.Serialization.SerializationInfo info,
            System.Runtime.Serialization.StreamingContext context)
            : base(info, context) { }
    }

    public class ModuleMap
    {
        public ModuleMap(string path)
        {
            this.m_Map = ReadModuleMap(path);
        }

        public int GetModuleId(string moduleName)
        {
            var ret = default(int);
            if (!m_Map.TryGetValue(moduleName, out ret))
            {
                throw new ModuleMapException(string.Format("モジュール名 '{0}' がモジュール番号マップファイルに存在しません。", moduleName));
            }
            return ret;
        }

        private IReadOnlyDictionary<string, int> m_Map;

        public static IReadOnlyDictionary<string, int> ReadModuleMap(string moduleMapFile)
        {
            if (moduleMapFile == null)
            {
                return null;
            }
            try
            {
                return GetModuleMap(File.ReadAllLines(moduleMapFile));
            }
            catch (Exception e)
            {
                var b = new StringBuilder();
                b.AppendLine(e.Message);
                b.AppendLine(string.Format(@"  モジュール変換マップファイルを確認してください : {0}", moduleMapFile));
                throw new Exception(b.ToString(), e);
            }
        }

        private static IReadOnlyDictionary<string, int> GetModuleMap(IEnumerable<string> lines)
        {
            var ret = new Dictionary<string, int>();
            var numbers = new HashSet<int>();
            foreach (var line in lines)
            {
                var kv = ParseLine(line);
                if (kv.HasValue)
                {
                    var key = kv.Value.Key;
                    var value = kv.Value.Value;
                    if (ret.ContainsKey(key))
                    {
                        throw new ModuleMapException(string.Format("モジュール番号マップファイルでキーの重複があります : {0}", key));
                    }
                    if (numbers.Contains(value))
                    {
                        throw new ModuleMapException(string.Format("モジュール番号マップファイルでモジュール値の重複があります : {0}", value));
                    }
                    ret.Add(key, value);
                    numbers.Add(value);
                }
            }
            return ret;
        }

        private static KeyValuePair<string, int>? ParseLine(string line)
        {
            line = DeleteComment(line, "//");
            line = DeleteComment(line, "#");
            line = line.Trim();
            if (line.Length == 0)
            {
                return null;
            }
            var words = line.Split('=');
            if (!(words.Length == 2))
            {
                throw new ModuleMapException(string.Format("モジュール番号マップファイルにエラーがあります : {0}", line));
            }
            var key = words[0].Trim();
            var valueString = words[1].Trim();
            int value;
            if (!int.TryParse(valueString, out value))
            {
                throw new ModuleMapException(string.Format("モジュール番号マップファイルにエラーがあります : {0}", line));
            }
            return new KeyValuePair<string, int>(key, value);
        }

        private static string DeleteComment(string line, string commentStarter)
        {
            var index = line.IndexOf(commentStarter);
            if (index >= 0)
            {
                line = line.Substring(0, index);
            }
            return line;
        }
    }
}
