﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nn.Adl.Syntax
{
    public interface ISyntaxInfo
    {
        IEnumerable<SyntaxElement> Children { get; }
        int Position { get; }
    }

    public class NonTerminalSyntaxInfo : ISyntaxInfo
    {
        public IEnumerable<SyntaxElement> Children { get; internal set; }
        public int Position
        {
            get
            {
                return Children.First().SyntaxInfo.Position;
            }
        }
    }

    public class TerminalSyntaxInfo : ISyntaxInfo
    {
        public string Text { get; set; }

        private static SyntaxElement[] s_Empty = new SyntaxElement[0];

        public IEnumerable<SyntaxElement> Children { get { return s_Empty;  } }
        public int Position { get; internal set; }
    }

    public class SyntaxElement
    {
        public ISyntaxInfo SyntaxInfo { get; internal set; }
    }

    // Terminal

    public abstract class TerminalElement : SyntaxElement
    {
    }

    public class EofElement : TerminalElement { }

    public class FixedElement<T> : TerminalElement
    {
        public T Value { get; set; }
        public override string ToString()
        {
            return string.Format(@"""{0}""", Value);
        }
    }

    public class Symbol : FixedElement<string> { }

    public class Keyword : FixedElement<string> { }

    public class Identifier : TerminalElement
    {
        public string Name { get; set; }
        public override string ToString()
        {
            return Name;
        }
    }

    public abstract class Literal : TerminalElement
    {
        public abstract object GetValue();
        public static Literal<T> Create<T>(T x) { return new Literal<T> { Value = x }; }
    }

    public class Literal<T> : Literal
    {
        public override object GetValue() { return Value; }
        public T Value { get; set; }
        public override string ToString()
        {
            return string.Format(@"{0}:'{1}'", typeof(T).Name, Value);
        }
    }

    // NonTerminal

    public abstract class NonTerminalElement : SyntaxElement { }

    public class EnumerableElement<T> : NonTerminalElement
    {
        public IEnumerable<T> Elements { get; private set; }
        public EnumerableElement(IEnumerable<T> e)
        {
            this.Elements = e;
        }
    }

    public class Document : NonTerminalElement
    {
        public EnumerableElement<Statement> Statements { get; set; }
    }

    public class Statement : NonTerminalElement
    {
        internal virtual void Fix() { }
    }

    public class Definition : Statement
    {
        public Identifier DefineeName { get; set; }
        public Expression ValueExpression { get; set; }
        public EnumerableElement<Modification> PreModifications { get; set; }
        public EnumerableElement<Modification> PostModifications { get; set; }
        public EnumerableElement<Statement> Children { get; set; }
        public IEnumerable<Modification> GetModifications()
        {
            return PreModifications.Elements.Concat(PostModifications.Elements);
        }

        private IEnumerable<Modification> m_TemporaryPreModifications = new Modification[0];
        private IEnumerable<Modification> m_TemporaryPostModifications = new Modification[0];

        internal void PushFront(IEnumerable<Modification> modification)
        {
            this.m_TemporaryPreModifications = modification.Concat(m_TemporaryPreModifications);
        }

        internal void PushBack(IEnumerable<Modification> modification)
        {
            this.m_TemporaryPostModifications = m_TemporaryPostModifications.Concat(modification);
        }

        internal override void Fix()
        {
            this.PreModifications = new EnumerableElement<Modification>(m_TemporaryPreModifications.ToArray());
            this.PostModifications = new EnumerableElement<Modification>(m_TemporaryPostModifications.ToArray());
            this.m_TemporaryPreModifications = null;
            this.m_TemporaryPostModifications = null;
        }
    }

    public class ScopedModification : Statement
    {
        public Modification[] Modifications { get; set; }
    }

    public class Group : Statement
    {
        public Modification[] Modifications { get; set; }
        public Statement[] Statements { get; set; }
    }

    public enum ModificationPosition
    {
        None,
        Scope,
        Group,
        PreModify,
        PostModify,
    }

    public abstract class Modification : NonTerminalElement
    {
        public ModificationPosition ModificationPosition { get; set; }
        public Identifier DefineeName { get; set; }
        public EnumerableElement<Identifier> Categories { get; set; }

        private IEnumerable<Identifier> m_TemporaryCategories;

        public void AddCategory(Identifier id)
        {
            if (m_TemporaryCategories == null)
            {
                this.m_TemporaryCategories = new[] { id };
            }
            else
            {
                this.m_TemporaryCategories = new[] { id }.Concat(m_TemporaryCategories);
            }
        }

        public void Fix()
        {
            if (m_TemporaryCategories != null)
            {
                this.Categories = new EnumerableElement<Identifier>(m_TemporaryCategories.ToArray());
            }
            this.m_TemporaryCategories = null;
        }
    }

    public class ValueModification : Modification
    {
        public Expression ValueExpression { get; set; }
    }

    public class NonValueModification : Modification
    {
        public Expression ValueExpression { get; set; }
    }

    public abstract class Expression : NonTerminalElement
    {
        public abstract object GetValue();
    }

    public class LiteralExpression : Expression
    {
        public Literal Literal { get; set; }
        public override string ToString() { return Literal.ToString(); }
        public override object GetValue() { return Literal.GetValue(); }
    }
}
