﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Nn.Adl.Syntax;

namespace Nn.Adl.Parsing
{
    internal class SyntaxErrorException : ApplicationException
    {
        public string[] Expected { get; private set; }
        public string Info { get; private set; }
        public string Detail { get; private set; }

        public SyntaxErrorException(string exp)
            : base("文法エラー")
        {
            Expected = exp.Split(' ');
        }

        public void SetInfo(string info, string detail)
        {
            Info = info;
            Detail = detail;
        }
    }

    internal class UnexpectedEndOfFileException : Exception
    {
    }

    public interface ITokenKindGetter<TokenKind>
        where TokenKind : struct
    {
        TokenKind? GetKeywordTokenKind(TokenValue tokenValue);
        TokenKind? GetSymbolTokenKind(TokenValue tokenValue);
        TokenKind GetIntegerTokenKind();
        TokenKind GetIdentifierTokenKind();
        TokenKind GetStringTokenKind();
        TokenKind GetEofTokenKind();
    }

    public interface IParser<TokenKind>
    {
        void reset();
        bool post(TokenKind token, object value, int pos);
        bool accept(out object v);
        dynamic GetParserStack();
        event System.Action<object, int, IEnumerable<object>> ReduceEventHandler;
    }

    public static class AutogenParserExtension
    {
        public static Document Parse<TokenKind>(this IParser<TokenKind> parser, DefaultTokenStream<TokenValue> ts, string initialPath, ITokenKindGetter<TokenKind> tokenKindGetter)
            where TokenKind : struct
        {
            parser.ReduceEventHandler += OnReduce;
            try
            {
                return ParsingImpl<TokenKind>.Parse(parser, ts, initialPath, tokenKindGetter);
            }
            finally
            {
                parser.ReduceEventHandler -= OnReduce;
            }
        }

        private static void OnReduce(object left, int pos, IEnumerable<object> rights)
        {
            var l = left as NonTerminalElement;
            if (l != null)
            {
                OnReduceImpl(l, pos, rights);
            }
        }

        private static void OnReduceImpl(NonTerminalElement left, int pos, IEnumerable<object> rights)
        {
            var list = new List<SyntaxElement>();
            AddSyntaxInfo(list, rights);
            left.SyntaxInfo = new NonTerminalSyntaxInfo { Children = list };
        }

        private static void AddSyntaxInfo(List<SyntaxElement> list, System.Collections.IEnumerable rights)
        {
            foreach (var right in rights)
            {
                var e = right as SyntaxElement;
                if (e != null)
                {
                    list.Add(e);
                }
                else
                {
                    AddSyntaxInfo(list, (System.Collections.IEnumerable)right);
                }
            }
        }
    }

    public static class ParsingUtility
    {
        public static string MakeExceptionMessage(string text, int position, Exception e, string path)
        {
            int errorPos = Math.Min(position, Math.Max(text.Length - 1, 0));

            // エラー行
            int lineBegin = text.LastIndexOfLineHead(errorPos);
            int lineEnd = text.IndexOfLineEnd(errorPos);
            string lineSample = text.Substring(lineBegin, lineEnd - lineBegin);

            string sample = " " + lineSample;

            // エラー行の前の行
            int preLineEnd = text.LastIndexOfLineEnd(errorPos);
            if (preLineEnd >= 0)
            {
                int preLineBegin = text.LastIndexOfLineHead(preLineEnd);
                string preLineSample = text.Substring(preLineBegin, preLineEnd - preLineBegin);

                sample = " " + preLineSample + "\n" + sample;
            }

            var tp = StringUtility.GetTextPoint(text, errorPos);
            int posInSample = tp.Chars;

            int numTab = lineSample.Substring(0, posInSample).Count(x => x == '\t');
            int tabOffset = numTab * (4 - 1);
            posInSample += tabOffset;

            sample = sample.Replace('\r', ' ');
            sample = sample.Replace("\t", "    ");

            string posMarker = new string(' ', posInSample) + "↑";

            var sb = new StringBuilder();

            sb.Append(e.Message)
                .Append(" at ").AppendLine(StringUtility.FormatTextPoint(path, tp, tabOffset))
                .AppendLine("  以下の「↑」の付近にエラーがあります。")
                .AppendLine("----------------------------------------")
                .AppendLine(sample)
                .AppendLine(posMarker)
                .AppendLine("----------------------------------------");

            return sb.ToString();
        }
    }

    internal static class ParsingImpl<TokenKind>
        where TokenKind : struct
    {
        private static string MakeParserStackString(dynamic stack, string text)
        {
            var sb = new StringBuilder();
            sb.AppendLine("パーサースタック:");

            for (int i = stack.get_stack_level() - 1; i >= 1; --i)
            {
                var sf = stack.get_arg(i, 0);
                var ttp = StringUtility.GetTextPoint(text, sf.pos);
                sb.AppendFormat("  line {0,4}, pos {1,3}: ", ttp.Lines, ttp.Chars);
                if (sf.token != null)
                {
                    sb.AppendLine(sf.token);
                }
                else
                {
                    sb.AppendLine("--");
                }
            }

            return sb.ToString();
        }

        private static string MakeExpectedString(string[] expected)
        {
            var sb = new StringBuilder();
            sb.AppendLine();
            sb.AppendLine("この場所には次のいずれかがあるべきです:");
            foreach (var s in expected)
            {
                sb.AppendFormat("  {0}\n", s);
            }
            return sb.ToString();
        }

        public static Document Parse(IParser<TokenKind> parser, DefaultTokenStream<TokenValue> ts, string initialPath, ITokenKindGetter<TokenKind> tokenKindGetter)
        {
            try
            {
                Parse(parser, ts, tokenKindGetter);
            }
            catch (InvalidCharacterException e)
            {
                throw new ErrorException(
                    ParsingUtility.MakeExceptionMessage(ts.Text, ts.Position, e, initialPath),
                    "認識できない文字です");
            }
            catch (SyntaxErrorException e)
            {
                throw new ErrorException(
                    ParsingUtility.MakeExceptionMessage(ts.Text, ts.Position, e, initialPath),
                    MakeParserStackString(parser.GetParserStack(), ts.Text)
                    + MakeExpectedString(e.Expected));
            }
            catch (UnexpectedEndOfFileException)
            {
                throw new ErrorException(
                    (new StringBuilder())
                        .AppendLine("文法エラー")
                        .AppendLine("  ファイルが途中で終わっています。")
                        .ToString());
            }

            object t;
            parser.accept(out t);
            return (Document)t;
        }

        private static void Parse(IParser<TokenKind> parser, DefaultTokenStream<TokenValue> ts, ITokenKindGetter<TokenKind> tokenKindGetter)
        {
            while (true)
            {
                var tv = ts.GetNextToken();
                TerminalElement element;
                var tokenKind = ReadToken(tv.Value, tokenKindGetter, out element);
                if (parser.post(tokenKind, element, tv.Position))
                {
                    break;
                }
            }
        }

        private static TokenKind ReadToken(TokenValue tv, ITokenKindGetter<TokenKind> tokenKindGetter, out TerminalElement element)
        {
            if (tv == null)
            {
                element = new EofElement();
                return tokenKindGetter.GetEofTokenKind();
            }
            else
            {
                var ret = ReadTokenImpl(tv, tokenKindGetter, out element);
                element.SyntaxInfo = new TerminalSyntaxInfo
                {
                    Text = tv.Text,
                    Position = tv.Position,
                };
                return ret;
            }
        }

        private static TokenKind ReadTokenImpl(TokenValue tv, ITokenKindGetter<TokenKind> tokenKindGetter, out TerminalElement element)
        {
            switch (tv.Type)
            {
                case TokenType.Number:
                    {
                        element = Literal.Create(long.Parse(tv.Text));
                        return tokenKindGetter.GetIntegerTokenKind();
                    }
                case TokenType.String:
                    {
                        element = Literal.Create(tv.Text);
                        return tokenKindGetter.GetStringTokenKind();
                    }
                case TokenType.Symbol:
                    {
                        var kind = tokenKindGetter.GetSymbolTokenKind(tv);
                        if (!kind.HasValue)
                        {
                            throw new ImplementationSupendedException();
                        }
                        element = new Symbol { Value = tv.Text };
                        return kind.Value;
                    }
                case TokenType.Word:
                    {
                        var kind = tokenKindGetter.GetKeywordTokenKind(tv);
                        if (kind.HasValue)
                        {
                            element = new Keyword { Value = tv.Text };
                            return kind.Value;
                        }
                        else
                        {
                            element = new Identifier { Name = tv.Text };
                            return tokenKindGetter.GetIdentifierTokenKind();
                        }
                    }
                default: throw new ImplementationSupendedException();
            }
        }
    }
}
