﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.ManuHostTools.UsbLibrary.Native;
using Microsoft.Win32.SafeHandles;
using Nintendo.ManuHostTools.UsbLibrary;

namespace Nintendo.ManuHostTools.UsbLibrary
{
    // ターゲットデバイス
    public class UsbSession : IDisposable
    {
        private bool disposed;
        private byte pipeBulkIn;
        private byte pipeBulkOut;
        private int interfaceNumber;
        private WinUsbHandle winUsbHandle;
        private UsbDevice usbDevice;

        public enum InterfaceNumber : uint
        {
            OldCommandRequest = 0,
            UsbFileIo = 1,
            UsbHostCommunicator = 2,
            CommandRequest = 3
        }

        public void CreateSession(UsbDevice device, int interfaceNumber)
        {
            this.usbDevice = device;

            if (!NativeMethods.WinUsb_Initialize(device.DeviceHandle, out this.winUsbHandle))
            {
                throw CreateExceptionFromWin32Error(Marshal.GetLastWin32Error());
            }

            var interfaceDescriptor = new NativeMethods.USB_INTERFACE_DESCRIPTOR();

            WinUsbHandle interfaceHandle = this.winUsbHandle;

            for (byte i=0; ;i++)
            {
                NativeMethods.WinUsb_QueryInterfaceSettings(interfaceHandle, 0, out interfaceDescriptor);
                if (interfaceDescriptor.bInterfaceNumber == interfaceNumber)
                {
                    this.interfaceNumber = interfaceNumber;
                    this.winUsbHandle = interfaceHandle;
                    break;
                }

                if (!NativeMethods.WinUsb_GetAssociatedInterface(this.winUsbHandle, i, out interfaceHandle))
                {
                    throw new InvalidDeviceException("Unable to find interfaceNumber : " + interfaceNumber + "\n");
                }
            }

            byte pipeBulkIn = 0;
            byte pipeBulkOut = 0;
            for (int i = 0; i < interfaceDescriptor.bNumEndpoints; i++)
            {
                NativeMethods.WINUSB_PIPE_INFORMATION pipeInfo;
                while (!NativeMethods.WinUsb_QueryPipe(this.winUsbHandle, (byte)0, (byte)i, out pipeInfo))
                {
                    Thread.Sleep(1);
                }

                if (pipeInfo.PipeType == NativeMethods.USBD_PIPE_TYPE.UsbdPipeTypeBulk && NativeMethods.USB_ENDPOINT_DIRECTION_IN(pipeInfo.PipeId))
                {
                    pipeBulkIn = pipeInfo.PipeId;
                }
                else if (pipeInfo.PipeType == NativeMethods.USBD_PIPE_TYPE.UsbdPipeTypeBulk && NativeMethods.USB_ENDPOINT_DIRECTION_OUT(pipeInfo.PipeId))
                {
                    pipeBulkOut = pipeInfo.PipeId;
                }

                Debug.WriteLine("PipeInfo: {0}, {1}, {2}, {3}", pipeInfo.PipeId, pipeInfo.PipeType, pipeInfo.Interval, pipeInfo.MaximumPacketSize);
            }

            if (pipeBulkIn == 0)
            {
                throw new InvalidDeviceException("BULK IN pipe was not found");
            }

            if (pipeBulkOut == 0)
            {
                throw new InvalidDeviceException("BULK OUT pipe was not found");
            }

            this.pipeBulkIn = pipeBulkIn;
            this.pipeBulkOut = pipeBulkOut;
        }

        ~UsbSession()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        public int Read(byte[] buffer, int offset, int count)
        {
            int countTransferred;
            if (!NativeMethods.WinUsb_ReadPipe(this.winUsbHandle, this.pipeBulkIn, buffer, offset, count, out countTransferred))
            {
                throw CreateExceptionFromWin32Error(Marshal.GetLastWin32Error());
            }

            return countTransferred;
        }

        public int Write(byte[] buffer, int offset, int count)
        {
            {
                uint value = 0;
                NativeMethods.WinUsb_SetPipePolicy(
                    this.winUsbHandle, this.pipeBulkOut, 3, Marshal.SizeOf(value), ref value);
            }
            int countTransferred;
            if (!NativeMethods.WinUsb_WritePipe(this.winUsbHandle, this.pipeBulkOut, buffer, offset, count, out countTransferred))
            {
                throw CreateExceptionFromWin32Error(Marshal.GetLastWin32Error());
            }

            return (int)countTransferred;
        }

        public void Abort()
        {
            if (!NativeMethods.WinUsb_AbortPipe(this.winUsbHandle, this.pipeBulkIn))
            {
                throw CreateExceptionFromWin32Error(Marshal.GetLastWin32Error());
            }

            if (!NativeMethods.WinUsb_AbortPipe(this.winUsbHandle, this.pipeBulkOut))
            {
                throw CreateExceptionFromWin32Error(Marshal.GetLastWin32Error());
            }
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    if (!this.usbDevice.DeviceHandle.IsInvalid)
                    {
                        this.usbDevice.DeviceHandle.Dispose();
                    }

                    if (!this.winUsbHandle.IsInvalid)
                    {
                        this.winUsbHandle.Dispose();
                    }
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        private static Exception CreateExceptionFromWin32Error(int errorCode)
        {
            var win32Exception = new Win32Exception();
            switch (errorCode)
            {
                case NativeMethods.Win32Error.ERROR_FILE_NOT_FOUND:   // デバイス切断時
                case NativeMethods.Win32Error.ERROR_INVALID_HANDLE:   // CreateFile した直後に切断された
                case NativeMethods.Win32Error.ERROR_BAD_COMMAND:      // デバイス切断時
                case NativeMethods.Win32Error.ERROR_GEN_FAILURE:      // デバイス切断時
                case NativeMethods.Win32Error.ERROR_OPERATION_ABORTED:
                case NativeMethods.Win32Error.ERROR_DEVICE_NOT_CONNECTED:
                    return new OperationCanceledException(win32Exception.Message, win32Exception);
                default:
                    return win32Exception;
            }
        }
    }
}
