﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Diagnostics;
using System.Reflection;

namespace MultipleRunFinalize
{
    partial class Program
    {
        static ProgramOption option;
        static void Main(string[] args)
        {
            try
            {
                option = ProgramOption.Parse(args);

                Console.WriteLine($"[Parameter]");
                Console.WriteLine($"          SdkRootPath : {option.SdkRootPath}");
                Console.WriteLine($"            BuildType : {option.BuildType}");
                Console.WriteLine($"             Platform : {option.Platform}");
                Console.WriteLine($"              KeyType : {option.KeyType}");
                Console.WriteLine($"             BootType : {option.BootType}");
                Console.WriteLine($"           SignedType : {option.SignedType}");
                Console.WriteLine($"          StorageSize : {option.StorageSize}");
                Console.WriteLine($"        DeveloperType : {option.DeveloperType}");
                Console.WriteLine($"           ConfigName : {option.ConfigName}");
                Console.WriteLine($"  FirstInitializeType : {option.FirstInitializeType}");
                Console.WriteLine($"       ConnectionType : {option.ConnectionType}");
                Console.WriteLine($"FirstConnectTargetNum : {option.FirstConnectTargetNum}");
                Console.WriteLine($"           LoggerPath : {option.LoggerPath}");
                Console.WriteLine("");
                TestEnsureExistDirectory(option.SdkRootPath);

                if(!File.Exists(option.LoggerPath))
                {
                    Console.WriteLine("[WARNING] Logger.exe が見つかりません。ログファイルを保存せずに実行します。");
                }

                CleanProcess();

                var amountStopWatch = Stopwatch.StartNew();
                var wrapStopWatch   = Stopwatch.StartNew();

                var deviceList = EnumerateUsbDeviceAtLeast("manu", option.FirstConnectTargetNum, 10);
                ExecuteMultiple(SystemInitializerBreakBctScript, deviceList);

                wrapStopWatch.Stop();
                Console.WriteLine("");
                Console.WriteLine($"                 BreakBct time (並列数 {deviceList.Length}) : {wrapStopWatch.Elapsed}");
                Console.WriteLine($"                             累計処理時間 : {amountStopWatch.Elapsed}");
                Console.WriteLine("");
                wrapStopWatch.Restart();

                /*
                deviceList = EnumerateUsbDeviceAtLeast("frcm", deviceList.Length, 10);
                ExecuteMultiple(WriteInvalidBootConfig, deviceList);

                wrapStopWatch.Stop();
                Console.WriteLine("");
                Console.WriteLine($"  WriteInvalidBootConfig time (並列数 {deviceList.Length})  : {wrapStopWatch.Elapsed}");
                Console.WriteLine($"                             累計処理時間 : {amountStopWatch.Elapsed}");
                Console.WriteLine("");
                wrapStopWatch.Restart();

                System.Threading.Thread.Sleep(TimeSpan.FromSeconds(4));
                */

                deviceList = EnumerateUsbDeviceAtLeast("frcm", deviceList.Length, 10);
                ExecuteMultiple(WriteByManuRcm, deviceList);

                wrapStopWatch.Stop();
                Console.WriteLine("");
                Console.WriteLine($"          WriteByManuRcm time (並列数 {deviceList.Length})  : {wrapStopWatch.Elapsed}");
                Console.WriteLine($"                             累計処理時間 : {amountStopWatch.Elapsed}");
                Console.WriteLine("");
                wrapStopWatch.Restart();

                deviceList = EnumerateUsbDeviceAtLeast("manu", deviceList.Length, 15);
                ExecuteMultiple(SystemInitializerManuScript, deviceList);
                System.Threading.Thread.Sleep(TimeSpan.FromSeconds(4));
                deviceList = EnumerateUsbDeviceAtLeast("manu", deviceList.Length, 10);

                wrapStopWatch.Stop();
                amountStopWatch.Stop();
                Console.WriteLine("");
                Console.WriteLine($"   SystemInitializerManu time (並列数 {deviceList.Length})  : {wrapStopWatch.Elapsed}");
                Console.WriteLine($"                             累計処理時間 : {amountStopWatch.Elapsed}");
                Console.WriteLine("");

            }
            catch(Exception e)
            {
                Console.WriteLine(e);
                Environment.Exit(1);
            }

            Environment.Exit(0);
        }

        private static void CleanProcess()
        {
            var wantedList = new string[] {
                "RunOnTargetFromNand",
                "RunSystemInitializerManu",
                "RequestToWsIssuer",
                "EnumerateUsbDevice",
                "UsbRecoveryImageWriter",
                "Logger",
            };

            foreach (var wanted in wantedList)
            {
                KillProcessByName(wanted);
            }
        }

        private static int WriteInvalidBootConfig(string devicePath = null)
        {
            string FunctionName = MethodBase.GetCurrentMethod().Name;
            string ScriptPath = Path.Combine(option.SdkRootPath, @"Integrate\Scripts\NX\Utility\Write-BootConfig-Frcm.ps1");
            string DummyBootConfigPath = Path.Combine(option.SdkRootPath, @"Externals\NxBootConfigs\bc-dummy.bin");
            string LogFileName = $"{FunctionName}_log_{DateTime.Now.ToString("yyyy_MM_dd-HH_mm_ss")}.txt";
            string TemporaryDirectory = Path.Combine(option.TemporaryDirectoryRoot, devicePath ?? String.Empty);

            CreateDirectoryIfNotExist(TemporaryDirectory);

            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Start");

            TestEnsureExistFile(ScriptPath);
            TestEnsureExistFile(DummyBootConfigPath);

            var process = InvokePowerShell(ScriptPath,
                $"-KeyType {option.KeyType} {DummyBootConfigPath}",
                Path.Combine(TemporaryDirectory, LogFileName));
            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Exit:{process.ExitCode}");

            return process.ExitCode;
        }

        private static int WriteByManuRcm(string devicePath = null)
        {
            string FunctionName = MethodBase.GetCurrentMethod().Name;
            string UsbRecoveryImageWriter = Path.Combine(option.SdkRootPath, @"Externals\NxSystemImages\UsbRecoveryImageWriter\UsbRecoveryImageWriter.exe");
            string UsbRecoveryImageWriterRcm = Path.Combine(option.SdkRootPath, $"Externals\\NxSystemImages\\UsbRecoveryImageWriter\\UsbRecoveryImageWriter.Signed.{option.KeyType}.rcm");
            string RecoveryWriterName = $"RecoveryWriter-{option.KeyType}-{option.ConnectionType}-{option.SignedType}";
            string RecoveryWriter = Path.Combine(option.SdkRootPath, $"Programs\\Eris\\Outputs\\{option.Platform}\\SystemImages\\QspiBootImages\\{RecoveryWriterName}\\{option.BuildType}\\{RecoveryWriterName}.qspi.img");
            string DevicePathOption = GetDevicePathOption(devicePath);
            string LogFileName = $"{FunctionName}_log_{DateTime.Now.ToString("yyyy_MM_dd-HH_mm_ss")}.txt";

            Console.WriteLine($"[{devicePath　??　"Unknown"}] {FunctionName} Start {LogFileName}");

            TestEnsureExistFile(UsbRecoveryImageWriter);
            TestEnsureExistFile(UsbRecoveryImageWriterRcm);
            TestEnsureExistFile(RecoveryWriter);

            var process = InvokeCommand(UsbRecoveryImageWriter, $"--rcm {UsbRecoveryImageWriterRcm} --image {RecoveryWriter} {DevicePathOption}", true);
            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Exit:{process.ExitCode}");

            string TemporaryDirectory = Path.Combine(option.TemporaryDirectoryRoot, devicePath);
            CreateDirectoryIfNotExist(TemporaryDirectory);
            File.WriteAllText(Path.Combine(TemporaryDirectory, LogFileName), process.StandardOutput.ReadToEnd());
            File.AppendAllText(Path.Combine(TemporaryDirectory, LogFileName), process.StandardError.ReadToEnd());

            return process.ExitCode;
        }

        private static int SystemInitializerManu(string devicePath = null)
        {
            string FunctionName = MethodBase.GetCurrentMethod().Name;
            string ImageName = $"NX-{option.KeyType}-{option.ConnectionType}-{option.SignedType}-{option.BootType}-{option.StorageSize}-{option.DeveloperType}";
            string ImagePath = Path.Combine(option.SdkRootPath, $"Programs\\Eris\\Outputs\\NX-NXFP2-a64\\InitialImages\\{ImageName}\\{option.BuildType}\\{ImageName}.initimgm");
            string DevicePathOption = GetDevicePathOption(devicePath);
            string TemporaryDirectory = Path.Combine(option.TemporaryDirectoryRoot, devicePath ?? String.Empty);

            CreateDirectoryIfNotExist(TemporaryDirectory);

            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Start");

            TestEnsureExistFile(option.RunSystemInitializerManu);
            TestEnsureExistFile(ImagePath);
            TestEnsureExistDirectory(option.TemporaryDirectoryRoot);
            TestEnsureExistDirectory(TemporaryDirectory);
            TestEnsureExistFile(option.ConfigPath);

            var process = InvokeCommand(option.RunSystemInitializerManu,
                $"{DevicePathOption} --verbose --image {ImagePath} --initializer 0100000000002103 --tmpdir {TemporaryDirectory} --config {option.ConfigPath} --timeout 180 -- --enable-overwriting-production-info --enable-erase-emmc",
                false);
            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Exit:{process.ExitCode}");

            return process.ExitCode;
        }

        private static int SystemInitializerManuScript(string devicePath = null)
        {
            string FunctionName = MethodBase.GetCurrentMethod().Name;
            string ScriptPath = Path.Combine(option.SdkRootPath, @"Integrate\Scripts\NX\Manufacturing\Manufacture-FinalizeWithAuth.ps1");
            string DevicePathOption = GetDevicePathOption(devicePath);
            string LogFileName = $"{FunctionName}_log_{DateTime.Now.ToString("yyyy_MM_dd-HH_mm_ss")}.txt";

            string TemporaryDirectory = Path.Combine(option.TemporaryDirectoryRoot, devicePath ?? String.Empty);
            CreateDirectoryIfNotExist(TemporaryDirectory);

            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Start {LogFileName}");

            TestEnsureExistFile(ScriptPath);

            var process = InvokePowerShell(ScriptPath,
                $"-BuildType {option.BuildType} -Platform {option.Platform} -KeyType {option.KeyType} -BootType {option.BootType} -SignedType {option.SignedType} -StorageSize {option.StorageSize} -DeveloperType {option.DeveloperType} -FirstInitializeType {option.FirstInitializeType} -ConfigName {option.ConfigName} -DevicePath \"{devicePath}\" -DoProcessClean 0",
                Path.Combine(TemporaryDirectory, LogFileName));
            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Exit:{process.ExitCode}");

            return process.ExitCode;
        }

        private static int SystemInitializerBreakBct(string devicePath = null)
        {
            string FunctionName = MethodBase.GetCurrentMethod().Name;
            string ImageName = "BreakBct";
            string ImagePath = Path.Combine(option.SdkRootPath, $"Programs\\Eris\\Outputs\\NX-NXFP2-a64\\InitialImages\\{ImageName}\\{option.BuildType}\\{ImageName}.{option.KeyType}.initimgm");
            string DevicePathOption = GetDevicePathOption(devicePath);
            string TemporaryDirectory = Path.Combine(option.TemporaryDirectoryRoot, devicePath ?? String.Empty);

            CreateDirectoryIfNotExist(TemporaryDirectory);

            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Start");

            TestEnsureExistFile(option.RunSystemInitializerManu);
            TestEnsureExistFile(ImagePath);
            TestEnsureExistDirectory(option.TemporaryDirectoryRoot);
            TestEnsureExistDirectory(TemporaryDirectory);
            TestEnsureExistFile(option.ConfigPath);

            var process = InvokeCommand(option.RunSystemInitializerManu,
                $"{DevicePathOption} --verbose --image {ImagePath} --initializer 0100000000002103 --tmpdir {TemporaryDirectory} --config {option.ConfigPath} --timeout 180 -- --enable-overwriting-production-info --enable-erase-emmc",
                false);
            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Exit:{process.ExitCode}");
            return process.ExitCode;
        }

        private static int SystemInitializerBreakBctScript(string devicePath = null)
        {
            string FunctionName = MethodBase.GetCurrentMethod().Name;
            string ScriptPath = Path.Combine(option.SdkRootPath, @"Integrate\Scripts\NX\Manufacturing\Manufacture-BreakBct.ps1");
            string DevicePathOption = GetDevicePathOption(devicePath);
            string LogFileName = $"{FunctionName}_log_{DateTime.Now.ToString("yyyy_MM_dd-HH_mm_ss")}.txt";

            string TemporaryDirectory = Path.Combine(option.TemporaryDirectoryRoot, devicePath ?? String.Empty);
            CreateDirectoryIfNotExist(TemporaryDirectory);

            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Start {LogFileName}");

            TestEnsureExistFile(ScriptPath);

            var process = InvokePowerShell(ScriptPath,
                $"-BuildType {option.BuildType} -Platform {option.Platform} -KeyType {option.KeyType} -ConfigName {option.ConfigName} -DevicePath \"{devicePath}\" -DoProcessClean 0",
                Path.Combine(TemporaryDirectory, LogFileName));
            Console.WriteLine($"[{devicePath ?? "Unknown"}] {FunctionName} Exit:{process.ExitCode}");

            return process.ExitCode;
        }

        private static String[] EnumerateUsbDevice(string type)
        {
            string EnumerateUsbDevice = Path.Combine(option.SdkRootPath, @"Tools\CommandLineTools\EnumerateUsbDevice.exe");

            TestEnsureExistFile(EnumerateUsbDevice);

            var process = InvokeCommand(EnumerateUsbDevice, $"--type {type}", true);
            var usbDevicePathList =  process.StandardOutput.ReadToEnd().Split('\n');

            usbDevicePathList = usbDevicePathList.Take(usbDevicePathList.Length - 1).ToArray();
            for (var i = 0; i < usbDevicePathList.Length; i++)
            {
                usbDevicePathList[i] = usbDevicePathList[i].Replace("\r","");
            }

            return usbDevicePathList;
        }

        private static String[] EnumerateUsbDeviceAtLeast(string type, int requireNum, int waitSeconds)
        {
            string[] devicePathList = {};

            var task = Task.Factory.StartNew(() => {
                int foundNum = 0;
                int count = 0;
                Console.WriteLine($"{requireNum} 個以上の{type}デバイスが見つかるまで待機…（タイムアウト:{waitSeconds}秒）");

                do
                {
                    devicePathList = EnumerateUsbDevice(type);
                    if (foundNum != devicePathList.Length)
                    {
                        foundNum = devicePathList.Length;
                        if (foundNum >= requireNum)
                        {
                            Console.WriteLine($"{foundNum} 個の{type}デバイスが見つかりました。処理を続行します。（待機時間:{count / 10}秒）");
                            break;
                        }
                    }

                    System.Threading.Thread.Sleep(TimeSpan.FromMilliseconds(100));
                    count++;

                    if (count > (waitSeconds * 10))
                    {
                        Console.WriteLine($"タイムアウトが発生しました。現在見つかっている {foundNum} 個に対して処理を続行します。");
                        break;
                    }
                }
                while (devicePathList.Length < requireNum);
            });

            task.Wait();

            return devicePathList;
        }

        delegate int ProcessFunction(string devicePath = null);
        private static int ExecuteMultiple(ProcessFunction function, String[] devicePathList)
        {
            var tasks = new Task[devicePathList.Length];

            for (int i=0; i< devicePathList.Length; i++)
            {
                var devicePath = devicePathList[i];
                tasks[i] = Task.Factory.StartNew(() => {
                    function(devicePath);
                });
            }

            Task.WaitAll(tasks);

            return 0;
        }
    }
}
